<?php
/**
 * Plugin Name: Adam's Backups
 * Description: Connect your site to Adam's managed backup service. Automatic offsite backups with a simple dashboard.
 * Version:     0.3.0
 * Author:      Adam / Dori Media
 * Text Domain: adams-backups
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'AB_BACKUPS_VERSION', '0.2.0' );
define( 'AB_BACKUPS_DIR', plugin_dir_path( __FILE__ ) );
define( 'AB_BACKUPS_URL', plugin_dir_url( __FILE__ ) );

/**
 * Default API base URL for the Adam's Backups SaaS.
 *
 * Update this once in the plugin code and all customer sites will connect
 * to the same backend. Customers do NOT need to change this.
 */
if ( ! defined( 'AB_BACKUPS_API_BASE' ) ) {
    define( 'AB_BACKUPS_API_BASE', 'https://apps.dorimedia.co.nz/api/v1' );
}

require_once AB_BACKUPS_DIR . 'includes/class-ab-settings.php';
require_once AB_BACKUPS_DIR . 'includes/class-ab-api-client.php';
require_once AB_BACKUPS_DIR . 'includes/class-ab-backup-runner.php';
require_once AB_BACKUPS_DIR . 'includes/class-ab-cron.php';

final class Adams_Backups_Plugin {

    private static $instance;

    /**
     * Singleton instance.
     *
     * @return Adams_Backups_Plugin
     */
    public static function instance() {
        if ( ! self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action( 'admin_menu', array( $this, 'add_menu' ) );
        add_action( 'admin_init', array( 'AB_Backups_Settings', 'register_settings' ) );

        // Cron hook is registered in AB_Backups_Cron; we just ensure it's loaded.
        add_action( 'ab_backups_run_scheduled_backup', array( $this, 'run_scheduled_backup' ) );
    }

    /**
     * Register admin menu item under Tools.
     */
    public function add_menu() {
        add_management_page(
            __( "Adam's Backups", 'adams-backups' ),
            __( "Adam's Backups", 'adams-backups' ),
            'manage_options',
            'adams-backups',
            array( $this, 'render_page' )
        );
    }

    /**
     * Render main plugin settings / status page.
     */
    public function render_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Manual backup trigger.
        $result = null;
        if ( isset( $_POST['ab_run_backup'] ) && check_admin_referer( 'ab_run_backup_action', 'ab_run_backup_nonce' ) ) {
            $runner = new AB_Backup_Runner();
            $result = $runner->run_backup_now();
        }

        $settings = AB_Backups_Settings::get_settings();
        $api      = new AB_Backups_API_Client();
        $backups  = null;
        $backups_error = null;

        if ( $api->is_configured() ) {
            $backups = $api->list_backups();
            if ( is_wp_error( $backups ) ) {
                $backups_error = $backups->get_error_message();
                $backups = null;
            } elseif ( isset( $backups['data'] ) && is_array( $backups['data'] ) ) {
                $backups = $backups['data'];
            }
        }
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( "Adam's Backups", 'adams-backups' ); ?></h1>

            <p>
                <?php esc_html_e( 'This plugin connects your site to your Adam’s Backups account. We will create backups on a schedule and upload them directly to offsite storage managed by you.', 'adams-backups' ); ?>
            </p>

            <?php if ( $result ) : ?>
                <div class="notice notice-<?php echo $result['success'] ? 'success' : 'error'; ?>">
                    <p><?php echo esc_html( $result['message'] ); ?></p>
                </div>
            <?php endif; ?>

            <h2><?php esc_html_e( 'Connection Settings', 'adams-backups' ); ?></h2>
            <p class="description">
                <?php esc_html_e( 'Enter the API token from your Adam’s Backups dashboard. The plugin already knows where to send your backups – you only need the token.', 'adams-backups' ); ?>
            </p>
            <?php AB_Backups_Settings::render_form(); ?>

            <hr />

            <h2><?php esc_html_e( 'Run a Backup Now', 'adams-backups' ); ?></h2>
            <p class="description">
                <?php esc_html_e( 'Use this to test your connection or grab an on-demand backup. Large sites may take a while, especially on shared hosting.', 'adams-backups' ); ?>
            </p>
            <form method="post">
                <?php wp_nonce_field( 'ab_run_backup_action', 'ab_run_backup_nonce' ); ?>
                <?php submit_button( __( 'Run Backup Now', 'adams-backups' ), 'primary', 'ab_run_backup' ); ?>
            </form>

            <hr />

            <h2><?php esc_html_e( 'Recent Backups', 'adams-backups' ); ?></h2>
            <p class="description">
                <?php esc_html_e( 'These are the backups stored safely offsite for this site. Status and size are reported by the backup service.', 'adams-backups' ); ?>
            </p>

            <?php if ( ! $api->is_configured() ) : ?>
                <p><?php esc_html_e( 'Enter your API token above and click “Save Changes” to see your backups.', 'adams-backups' ); ?></p>
            <?php elseif ( $backups_error ) : ?>
                <div class="notice notice-error">
                    <p><?php echo esc_html( $backups_error ); ?></p>
                </div>
            <?php elseif ( empty( $backups ) ) : ?>
                <p><?php esc_html_e( 'No backups found yet. Run a backup above or wait for the scheduled backup to run.', 'adams-backups' ); ?></p>
            <?php else : ?>
                <table class="widefat fixed striped">
                    <thead>
                    <tr>
                        <th><?php esc_html_e( 'Created', 'adams-backups' ); ?></th>
                        <th><?php esc_html_e( 'Type', 'adams-backups' ); ?></th>
                        <th><?php esc_html_e( 'Status', 'adams-backups' ); ?></th>
                        <th><?php esc_html_e( 'Size', 'adams-backups' ); ?></th>
                        <th><?php esc_html_e( 'Completed', 'adams-backups' ); ?></th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach ( $backups as $b ) : ?>
                        <tr>
                            <td><?php echo isset( $b['created_at'] ) ? esc_html( $b['created_at'] ) : '—'; ?></td>
                            <td><?php echo isset( $b['type'] ) ? esc_html( ucfirst( $b['type'] ) ) : '—'; ?></td>
                            <td><?php echo isset( $b['status'] ) ? esc_html( ucfirst( $b['status'] ) ) : '—'; ?></td>
                            <td>
                                <?php
                                if ( isset( $b['size_bytes'] ) && is_numeric( $b['size_bytes'] ) ) {
                                    $size = (int) $b['size_bytes'];
                                    if ( $size > 1024 * 1024 * 1024 ) {
                                        printf( '%.2f GB', $size / ( 1024 * 1024 * 1024 ) );
                                    } elseif ( $size > 1024 * 1024 ) {
                                        printf( '%.2f MB', $size / ( 1024 * 1024 ) );
                                    } elseif ( $size > 1024 ) {
                                        printf( '%.2f KB', $size / 1024 );
                                    } else {
                                        printf( '%d B', $size );
                                    }
                                } else {
                                    echo '—';
                                }
                                ?>
                            </td>
                            <td><?php echo isset( $b['completed_at'] ) ? esc_html( $b['completed_at'] ) : '—'; ?></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Cron callback wrapper.
     */
    public function run_scheduled_backup() {
        $runner = new AB_Backup_Runner();
        $runner->run_backup_now();
    }
}

// Bootstrap plugin.
add_action( 'plugins_loaded', array( 'Adams_Backups_Plugin', 'instance' ) );

// Activation / deactivation hooks for setting up cron.
register_activation_hook( __FILE__, array( 'AB_Backups_Cron', 'activate' ) );
register_deactivation_hook( __FILE__, array( 'AB_Backups_Cron', 'deactivate' ) );
