<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Simple HTTP API client for Adam's Backups SaaS.
 */
class AB_Backups_API_Client {

    /**
     * @var string
     */
    protected $base_url;

    /**
     * @var string
     */
    protected $api_token;

    /**
     * Constructor.
     *
     * @param string|null $api_token Optional explicit API token. If omitted, the saved setting is used.
     */
    public function __construct( $api_token = null ) {
        $this->base_url = trailingslashit( AB_BACKUPS_API_BASE );

        if ( null === $api_token ) {
            if ( class_exists( 'AB_Backups_Settings' ) ) {
                $settings  = AB_Backups_Settings::get_settings();
                $api_token = isset( $settings['api_token'] ) ? $settings['api_token'] : '';
            } else {
                $api_token = '';
            }
        }

        $this->api_token = trim( (string) $api_token );
    }

    /**
     * Whether the client has an API token configured.
     *
     * @return bool
     */
    public function is_configured() {
        return ! empty( $this->api_token );
    }

    /**
     * Perform a HTTP request to the backup API.
     *
     * @param string       $method         HTTP method.
     * @param string       $path           Path relative to the API base.
     * @param array|null   $body           Optional JSON body.
     * @param bool         $authenticated  Whether to send the Bearer token header.
     *
     * @return array|WP_Error
     */
    protected function request( $method, $path, $body = null, $authenticated = true ) {
        $url  = trailingslashit( $this->base_url ) . ltrim( $path, '/' );

        $args = array(
            'method'  => strtoupper( $method ),
            'timeout' => 20,
            'headers' => array(
                'Accept' => 'application/json',
            ),
        );

        if ( $authenticated && $this->api_token ) {
            $args['headers']['Authorization'] = 'Bearer ' . $this->api_token;
        }

        if ( null !== $body ) {
            $args['headers']['Content-Type'] = 'application/json';
            $args['body']                     = wp_json_encode( $body );
        }

        $response = wp_remote_request( $url, $args );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $raw  = wp_remote_retrieve_body( $response );
        $json = json_decode( $raw, true );

        if ( $code < 200 || $code >= 300 ) {
            return new WP_Error(
                'ab_bad_response',
                sprintf( __( 'Backup API returned HTTP %d', 'adams-backups' ), $code ),
                array( 'body' => $raw )
            );
        }

        if ( null === $json ) {
            $json = array();
        }

        return $json;
    }

    /**
     * Register this site and start a free trial.
     *
     * This endpoint does NOT require authentication.
     *
     * @param string $email     Admin/customer email.
     * @param string $site_name Site name.
     * @param string $customer_name Optional customer display name.
     *
     * @return array|WP_Error
     */
    public function register_site( $email, $site_name, $customer_name = '' ) {
        $payload = array(
            'site_url'      => get_site_url(),
            'site_name'     => $site_name,
            'admin_email'   => $email,
            'customer_name' => $customer_name ? $customer_name : $site_name,
        );

        return $this->request( 'POST', 'sites/register', $payload, false );
    }

    /**
     * Fetch remote config / plan / billing info for this site.
     *
     * @return array|WP_Error
     */
    public function get_config() {
        if ( ! $this->is_configured() ) {
            return new WP_Error(
                'ab_not_configured',
                __( 'Adam’s Backups is not configured with an API token.', 'adams-backups' )
            );
        }

        return $this->request( 'GET', 'config', null, true );
    }

    /**
     * Fetch a Stripe customer portal URL so the user can manage billing.
     *
     * @return array|WP_Error Returns array with key "url" on success.
     */
    public function get_billing_portal_url() {
        if ( ! $this->is_configured() ) {
            return new WP_Error(
                'ab_not_configured',
                __( 'Adam’s Backups is not configured with an API token.', 'adams-backups' )
            );
        }

        return $this->request( 'POST', 'billing/portal-url', array(), true );
    }

    /**
     * Start a backup.
     *
     * @param string     $type                 Backup type (e.g. full).
     * @param int|null   $estimated_size_bytes Optional rough estimate of site size.
     *
     * @return array|WP_Error
     */
    public function start_backup( $type = 'full', $estimated_size_bytes = null ) {
        if ( ! $this->is_configured() ) {
            return new WP_Error(
                'ab_not_configured',
                __( 'Adam’s Backups is not configured with an API token.', 'adams-backups' )
            );
        }

        $payload = array(
            'type' => $type,
        );

        if ( null !== $estimated_size_bytes ) {
            $payload['estimated_size_bytes'] = (int) $estimated_size_bytes;
        }

        return $this->request( 'POST', 'backups/start', $payload, true );
    }

    /**
     * Mark a backup as completed (success or failure).
     *
     * @param string      $backup_id
     * @param string      $status             success|failed
     * @param int|null    $size_bytes
     * @param int|null    $duration_seconds
     * @param string|null $error_message
     *
     * @return array|WP_Error
     */
    public function complete_backup( $backup_id, $status, $size_bytes = null, $duration_seconds = null, $error_message = null ) {
        if ( ! $this->is_configured() ) {
            return new WP_Error(
                'ab_not_configured',
                __( 'Adam’s Backups is not configured with an API token.', 'adams-backups' )
            );
        }

        $payload = array(
            'backup_id' => $backup_id,
            'status'    => $status,
        );

        if ( null !== $size_bytes ) {
            $payload['size_bytes'] = (int) $size_bytes;
        }

        if ( null !== $duration_seconds ) {
            $payload['duration_seconds'] = (int) $duration_seconds;
        }

        if ( null !== $error_message ) {
            $payload['error_message'] = (string) $error_message;
        }

        return $this->request( 'POST', 'backups/complete', $payload, true );
    }

    /**
     * List recent backups for this site.
     *
     * @return array|WP_Error
     */
    public function list_backups() {
        if ( ! $this->is_configured() ) {
            return new WP_Error(
                'ab_not_configured',
                __( 'Adam’s Backups is not configured with an API token.', 'adams-backups' )
            );
        }

        return $this->request( 'GET', 'backups', null, true );
    }
}

