<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Main backup runner.
 *
 * NOTE: This is a simple implementation designed for small to medium sites.
 * Very large sites may need a more advanced approach (chunked exports, WP-CLI).
 */
class AB_Backup_Runner {

    /**
     * Run a backup immediately.
     *
     * @return array { success: bool, message: string }
     */
    public function run_backup_now() {
        $start_time = microtime( true );
        $api        = new AB_Backups_API_Client();

        if ( ! $api->is_configured() ) {
            return array(
                'success' => false,
                'message' => __( 'Adam’s Backups is not configured yet. Please enter your API token first.', 'adams-backups' ),
            );
        }

        // Estimate size (rough: uploads dir only).
        $uploads     = wp_get_upload_dir();
        $uploads_dir = isset( $uploads['basedir'] ) ? $uploads['basedir'] : '';
        $estimated   = $uploads_dir ? $this->estimate_directory_size( $uploads_dir, 50 * 1024 * 1024 ) : null; // cap for speed.

        $start = $api->start_backup( 'full', $estimated );
        if ( is_wp_error( $start ) ) {
            return array(
                'success' => false,
                'message' => sprintf(
                    /* translators: %s: error message */
                    __( 'Failed to start backup: %s', 'adams-backups' ),
                    $start->get_error_message()
                ),
            );
        }

        if ( empty( $start['backup_id'] ) || empty( $start['upload'] ) || empty( $start['upload']['url'] ) ) {
            return array(
                'success' => false,
                'message' => __( 'Backup API did not return upload details. Check your Adam’s Backups API settings.', 'adams-backups' ),
            );
        }

        $backup_id = $start['backup_id'];
        $upload    = $start['upload'];

        // Prepare temp working directory.
        $upload_dir = wp_get_upload_dir();
        $base_temp  = trailingslashit( $upload_dir['basedir'] ) . 'adams-backups-temp';

        if ( ! wp_mkdir_p( $base_temp ) ) {
            $api->complete_backup( $backup_id, 'failed', null, null, 'Could not create temp directory.' );

            return array(
                'success' => false,
                'message' => __( 'Could not create temporary directory for backup.', 'adams-backups' ),
            );
        }

        $backup_slug = 'backup-' . gmdate( 'Ymd-His' ) . '-' . wp_generate_password( 6, false );
        $work_dir    = trailingslashit( $base_temp ) . $backup_slug;

        if ( ! wp_mkdir_p( $work_dir ) ) {
            $api->complete_backup( $backup_id, 'failed', null, null, 'Could not create working directory.' );

            return array(
                'success' => false,
                'message' => __( 'Could not create working directory for backup.', 'adams-backups' ),
            );
        }

        $zip_path = $work_dir . '.zip';

        $error_msg  = null;
        $size_bytes = 0;

        try {
            // 1) Export database.
            $db_file = trailingslashit( $work_dir ) . 'db.sql';
            $this->export_database_to_file( $db_file );

            // 2) Create ZIP archive (db + uploads).
            $size_bytes = $this->create_zip_archive( $zip_path, $db_file, $uploads_dir );

            // 3) Upload ZIP to pre-signed URL.
            $upload_result = $this->upload_zip( $zip_path, $upload );

            if ( is_wp_error( $upload_result ) ) {
                $error_msg = $upload_result->get_error_message();
            }
        } catch ( Exception $e ) {
            $error_msg = $e->getMessage();
        }

        // Clean up temp files.
        $this->cleanup_directory( $work_dir );
        if ( file_exists( $zip_path ) ) {
            @unlink( $zip_path );
        }

        $duration = (int) round( microtime( true ) - $start_time );

        if ( $error_msg ) {
            $api->complete_backup( $backup_id, 'failed', $size_bytes ?: null, $duration, $error_msg );

            return array(
                'success' => false,
                'message' => sprintf(
                    /* translators: %s: error message */
                    __( 'Backup failed: %s', 'adams-backups' ),
                    $error_msg
                ),
            );
        }

        $api->complete_backup( $backup_id, 'success', $size_bytes, $duration );

        return array(
            'success' => true,
            'message' => __( 'Backup created and uploaded successfully.', 'adams-backups' ),
        );
    }

    /**
     * Rough directory size estimate (may cap early).
     *
     * @param string $path
     * @param int    $limit
     * @return int
     */
    protected function estimate_directory_size( $path, $limit ) {
        $total = 0;

        if ( ! is_dir( $path ) ) {
            return 0;
        }

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $path, FilesystemIterator::SKIP_DOTS )
        );

        foreach ( $iterator as $file ) {
            if ( $file->isFile() ) {
                $total += (int) $file->getSize();
                if ( $total > $limit ) {
                    break;
                }
            }
        }

        return $total;
    }

    /**
     * Export WordPress database to a .sql file (simple implementation).
     *
     * @param string $file_path
     */
    protected function export_database_to_file( $file_path ) {
        global $wpdb;

        $tables = $wpdb->get_col( 'SHOW TABLES' );
        if ( empty( $tables ) ) {
            throw new Exception( 'No database tables found.' );
        }

        $fh = fopen( $file_path, 'w' );
        if ( ! $fh ) {
            throw new Exception( 'Unable to create database dump file.' );
        }

        fwrite( $fh, '-- Adam\'s Backups SQL export' . "\n" );
        fwrite( $fh, '-- Generated: ' . gmdate( 'c' ) . "\n\n" );

        foreach ( $tables as $table ) {
            // Drop + create.
            fwrite( $fh, 'DROP TABLE IF EXISTS `' . esc_sql( $table ) . '`;' . "\n" );

            $create = $wpdb->get_row( 'SHOW CREATE TABLE `' . $table . '`', ARRAY_N );
            if ( isset( $create[1] ) ) {
                fwrite( $fh, $create[1] . ';' . "\n\n" );
            }

            // Data rows.
            $rows = $wpdb->get_results( 'SELECT * FROM `' . $table . '`', ARRAY_A );
            if ( ! empty( $rows ) ) {
                foreach ( $rows as $row ) {
                    $values = array();
                    foreach ( $row as $value ) {
                        if ( null === $value ) {
                            $values[] = 'NULL';
                        } else {
                            $values[] = "'" . esc_sql( $value ) . "'"; // basic escape
                        }
                    }
                    fwrite(
                        $fh,
                        'INSERT INTO `' . $table . '` VALUES(' . implode( ',', $values ) . ');' . "\n"
                    );
                }
                fwrite( $fh, "\n" );
            }
        }

        fclose( $fh );
    }

    /**
     * Create a ZIP archive containing DB dump and uploads directory.
     *
     * @param string      $zip_path
     * @param string      $db_file
     * @param string|null $uploads_dir
     *
     * @return int Size of resulting ZIP in bytes
     */
    protected function create_zip_archive( $zip_path, $db_file, $uploads_dir = null ) {
        if ( ! class_exists( 'ZipArchive' ) ) {
            throw new Exception( 'ZipArchive PHP extension is not available.' );
        }

        $zip = new ZipArchive();
        if ( true !== $zip->open( $zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE ) ) {
            throw new Exception( 'Unable to create ZIP archive.' );
        }

        // Add DB file.
        if ( file_exists( $db_file ) ) {
            $zip->addFile( $db_file, 'db.sql' );
        }

        // Add uploads directory.
        if ( $uploads_dir && is_dir( $uploads_dir ) ) {
            $this->add_folder_to_zip( $zip, $uploads_dir, 'uploads' );
        }

        $zip->close();

        return file_exists( $zip_path ) ? (int) filesize( $zip_path ) : 0;
    }

    /**
     * Recursively add a folder to a ZipArchive.
     *
     * @param ZipArchive $zip
     * @param string     $folder_path
     * @param string     $zip_path
     */
    protected function add_folder_to_zip( ZipArchive $zip, $folder_path, $zip_path ) {
        $folder_path = rtrim( $folder_path, DIRECTORY_SEPARATOR );

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $folder_path, FilesystemIterator::SKIP_DOTS )
        );

        $folder_len = strlen( $folder_path );

        foreach ( $iterator as $file ) {
            if ( ! $file->isFile() ) {
                continue;
            }

            $file_path  = $file->getPathname();
            $local_path = $zip_path . substr( $file_path, $folder_len );

            // Normalise slashes.
            $local_path = ltrim( str_replace( DIRECTORY_SEPARATOR, '/', $local_path ), '/' );

            $zip->addFile( $file_path, $local_path );
        }
    }

    /**
     * Upload backup ZIP to the pre-signed URL.
     *
     * @param string $zip_path
     * @param array  $upload
     *
     * @return true|WP_Error
     */
    protected function upload_zip( $zip_path, $upload ) {
        if ( ! file_exists( $zip_path ) ) {
            return new WP_Error( 'ab_missing_zip', __( 'Backup ZIP file not found.', 'adams-backups' ) );
        }

        if ( empty( $upload['url'] ) ) {
            return new WP_Error( 'ab_missing_url', __( 'Upload URL missing from API response.', 'adams-backups' ) );
        }

        $method  = isset( $upload['method'] ) ? strtoupper( $upload['method'] ) : 'PUT';
        $headers = isset( $upload['headers'] ) && is_array( $upload['headers'] ) ? $upload['headers'] : array();

        // Ensure at least a Content-Type.
        if ( empty( $headers['Content-Type'] ) ) {
            $headers['Content-Type'] = 'application/zip';
        }

        $args = array(
            'method'  => $method,
            'headers' => $headers,
            'timeout' => 600, // allow large uploads
            'body'    => file_get_contents( $zip_path ),
        );

        $response = wp_remote_request( $upload['url'], $args );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( $code >= 200 && $code < 300 ) {
            return true;
        }

        return new WP_Error(
            'ab_upload_failed',
            sprintf(
                /* translators: %d: HTTP status code */
                __( 'Upload failed with HTTP status %d.', 'adams-backups' ),
                $code
            )
        );
    }

    /**
     * Recursively remove a directory.
     *
     * @param string $dir
     */
    protected function cleanup_directory( $dir ) {
        if ( ! is_dir( $dir ) ) {
            return;
        }

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $dir, FilesystemIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ( $iterator as $file ) {
            if ( $file->isDir() ) {
                @rmdir( $file->getPathname() );
            } else {
                @unlink( $file->getPathname() );
            }
        }

        @rmdir( $dir );
    }
}
