<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Cron helper for scheduling backups.
 */
class AB_Backups_Cron {

    const HOOK = 'ab_backups_run_scheduled_backup';

    /**
     * Schedule backup event based on settings on activation.
     */
    public static function activate() {
        if ( ! class_exists( 'AB_Backups_Settings' ) ) {
            return;
        }

        $settings = AB_Backups_Settings::get_settings();
        $freq     = isset( $settings['schedule_frequency'] ) ? $settings['schedule_frequency'] : 'daily';

        self::reschedule( $freq );
    }

    /**
     * Unschedule on deactivation.
     */
    public static function deactivate() {
        $timestamp = wp_next_scheduled( self::HOOK );
        while ( $timestamp ) {
            wp_unschedule_event( $timestamp, self::HOOK );
            $timestamp = wp_next_scheduled( self::HOOK );
        }
    }

    /**
     * Reschedule cron according to selected frequency.
     *
     * @param string $frequency daily|weekly|off
     */
    public static function reschedule( $frequency ) {
        // Clear any existing events.
        self::deactivate();

        if ( 'off' === $frequency ) {
            return;
        }

        $timestamp = strtotime( 'tomorrow 3:15am' );
        if ( ! $timestamp ) {
            $timestamp = time() + DAY_IN_SECONDS;
        }

        if ( 'weekly' === $frequency ) {
            $recurrence = 'weekly';
            // Ensure 'weekly' exists.
            add_filter(
                'cron_schedules',
                function ( $schedules ) {
                    if ( ! isset( $schedules['weekly'] ) ) {
                        $schedules['weekly'] = array(
                            'interval' => 7 * DAY_IN_SECONDS,
                            'display'  => __( 'Once Weekly', 'adams-backups' ),
                        );
                    }
                    return $schedules;
                }
            );
        } else {
            $recurrence = 'daily';
        }

        wp_schedule_event( $timestamp, $recurrence, self::HOOK );
    }
}
