<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Settings handler for Adam's Backups.
 */
class AB_Backups_Settings {

    const OPTION_KEY = 'ab_backups_settings';

    /**
     * Get merged settings with defaults.
     *
     * @return array
     */
    public static function get_settings() {
        $defaults = array(
            'api_token'          => '',
            'schedule_frequency' => 'daily',
        );

        $saved = get_option( self::OPTION_KEY, array() );
        if ( ! is_array( $saved ) ) {
            $saved = array();
        }

        return wp_parse_args( $saved, $defaults );
    }

    /**
     * Register settings with the WordPress Settings API.
     */
    public static function register_settings() {
        register_setting(
            'ab_backups_settings_group',
            self::OPTION_KEY,
            array(
                'type'              => 'array',
                'sanitize_callback' => array( __CLASS__, 'sanitize_settings' ),
            )
        );
    }

    /**
     * Sanitize options before saving.
     *
     * @param array $input Raw input.
     *
     * @return array
     */
    public static function sanitize_settings( $input ) {
        $output = self::get_settings();
        $input  = is_array( $input ) ? $input : array();

        if ( isset( $input['api_token'] ) ) {
            $output['api_token'] = sanitize_text_field( $input['api_token'] );
        }

        if ( isset( $input['schedule_frequency'] ) ) {
            $allowed = array( 'daily', 'weekly', 'off' );
            $value   = sanitize_text_field( $input['schedule_frequency'] );
            if ( in_array( $value, $allowed, true ) ) {
                $output['schedule_frequency'] = $value;
            }
        }

        return $output;
    }

    /**
     * Render settings form & onboarding / billing helpers.
     */
    public static function render_form() {
        $messages = array();
        $errors   = array();

        // Load current settings.
        $settings = self::get_settings();

        // Handle "Start trial" onboarding.
        if ( isset( $_POST['ab_backups_start_trial'] ) && check_admin_referer( 'ab_backups_start_trial', 'ab_backups_start_trial_nonce' ) ) {
            $email     = isset( $_POST['ab_trial_email'] ) ? sanitize_email( wp_unslash( $_POST['ab_trial_email'] ) ) : '';
            $site_name = isset( $_POST['ab_trial_site_name'] ) ? sanitize_text_field( wp_unslash( $_POST['ab_trial_site_name'] ) ) : get_bloginfo( 'name' );

            if ( empty( $email ) || ! is_email( $email ) ) {
                $errors[] = __( 'Please enter a valid email address.', 'adams-backups' );
            } else {
                $client = new AB_Backups_API_Client( '' );
                $resp   = $client->register_site( $email, $site_name );

                if ( is_wp_error( $resp ) ) {
                    $errors[] = $resp->get_error_message();
                } else {
                    if ( ! empty( $resp['api_token'] ) ) {
                        $settings['api_token'] = sanitize_text_field( $resp['api_token'] );
                        update_option( self::OPTION_KEY, $settings );
                        $messages[] = __( 'Your site has been connected to Adam’s Backups and a subscription has been created.', 'adams-backups' );
                    } else {
                        $errors[] = __( 'The backup service did not return an API token. Please contact support.', 'adams-backups' );
                    }
                }
            }

            // Reload settings after possible update.
            $settings = self::get_settings();
        }

        // Handle "Manage billing" portal redirect.
        if ( isset( $_POST['ab_backups_manage_billing'] ) && check_admin_referer( 'ab_backups_manage_billing', 'ab_backups_manage_billing_nonce' ) ) {
            $client = new AB_Backups_API_Client();
            $resp   = $client->get_billing_portal_url();

            if ( is_wp_error( $resp ) ) {
                $errors[] = $resp->get_error_message();
            } elseif ( ! empty( $resp['url'] ) ) {
                wp_safe_redirect( $resp['url'] );
                exit;
            } else {
                $errors[] = __( 'Could not get a billing portal URL. Please try again or contact support.', 'adams-backups' );
            }
        }

        // Optionally fetch config / plan info when connected.
        $config       = null;
        $config_error = null;

        if ( ! empty( $settings['api_token'] ) ) {
            $client = new AB_Backups_API_Client();
            $cfg    = $client->get_config();

            if ( is_wp_error( $cfg ) ) {
                $config_error = $cfg->get_error_message();
            } else {
                $config = $cfg;
            }
        }

        // Show notices.
        foreach ( $messages as $msg ) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html( $msg ) . '</p></div>';
        }

        foreach ( $errors as $msg ) {
            echo '<div class="notice notice-error"><p>' . esc_html( $msg ) . '</p></div>';
        }

        // If there is no API token yet, show the onboarding / trial form instead of the normal settings.
        if ( empty( $settings['api_token'] ) ) {
            $default_email = get_option( 'admin_email' );
            $default_name  = get_bloginfo( 'name' );
            ?>
            <form method="post">
                <?php wp_nonce_field( 'ab_backups_start_trial', 'ab_backups_start_trial_nonce' ); ?>

                <h3><?php esc_html_e( 'Connect this site to Adam’s Backups', 'adams-backups' ); ?></h3>
                <p class="description">
                    <?php esc_html_e( 'We’ll create an account for you, start a subscription, and link this WordPress site to your backup dashboard.', 'adams-backups' ); ?>
                </p>

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">
                            <label for="ab_trial_email"><?php esc_html_e( 'Your email address', 'adams-backups' ); ?></label>
                        </th>
                        <td>
                            <input type="email"
                                   class="regular-text"
                                   id="ab_trial_email"
                                   name="ab_trial_email"
                                   value="<?php echo esc_attr( $default_email ); ?>"
                                   required />
                            <p class="description">
                                <?php esc_html_e( 'We’ll use this for your backup account and important billing or backup notifications.', 'adams-backups' ); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="ab_trial_site_name"><?php esc_html_e( 'Site name', 'adams-backups' ); ?></label>
                        </th>
                        <td>
                            <input type="text"
                                   class="regular-text"
                                   id="ab_trial_site_name"
                                   name="ab_trial_site_name"
                                   value="<?php echo esc_attr( $default_name ); ?>" />
                            <p class="description">
                                <?php esc_html_e( 'This helps you recognise the site in your Adam’s Backups dashboard.', 'adams-backups' ); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <?php submit_button( __( 'Start backup subscription', 'adams-backups' ), 'primary', 'ab_backups_start_trial' ); ?>
            </form>
            <?php
            return;
        }

        // Connected state: show token + schedule + billing button.
        ?>
        <form method="post" action="options.php">
            <?php
            settings_fields( 'ab_backups_settings_group' );
            ?>

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label for="ab_api_token"><?php esc_html_e( 'API Token', 'adams-backups' ); ?></label>
                    </th>
                    <td>
                        <input name="<?php echo esc_attr( self::OPTION_KEY ); ?>[api_token]"
                               id="ab_api_token"
                               type="text"
                               class="regular-text"
                               value="<?php echo esc_attr( $settings['api_token'] ); ?>" />
                        <p class="description">
                            <?php esc_html_e( 'Per-site secret token from your Adam’s Backups dashboard. Treat this like a password.', 'adams-backups' ); ?>
                        </p>

                        <?php if ( $config ) : ?>
                            <p>
                                <strong><?php esc_html_e( 'Status:', 'adams-backups' ); ?></strong>
                                <?php
                                $plan  = isset( $config['plan'] ) ? $config['plan'] : '';
                                $state = isset( $config['billing_status'] ) ? $config['billing_status'] : '';

                                if ( $plan ) {
                                    printf(
                                        '%1$s — %2$s',
                                        esc_html( ucfirst( $plan ) ),
                                        esc_html( $state ? ucfirst( $state ) : __( 'Active', 'adams-backups' ) )
                                    );
                                } elseif ( $state ) {
                                    echo esc_html( ucfirst( $state ) );
                                }
                                ?>
                            </p>
                        <?php elseif ( $config_error ) : ?>
                            <p class="description">
                                <?php echo esc_html( $config_error ); ?>
                            </p>
                        <?php endif; ?>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="ab_schedule_frequency"><?php esc_html_e( 'Backup schedule', 'adams-backups' ); ?></label>
                    </th>
                    <td>
                        <select name="<?php echo esc_attr( self::OPTION_KEY ); ?>[schedule_frequency]" id="ab_schedule_frequency">
                            <option value="daily" <?php selected( $settings['schedule_frequency'], 'daily' ); ?>>
                                <?php esc_html_e( 'Daily (recommended)', 'adams-backups' ); ?>
                            </option>
                            <option value="weekly" <?php selected( $settings['schedule_frequency'], 'weekly' ); ?>>
                                <?php esc_html_e( 'Weekly', 'adams-backups' ); ?>
                            </option>
                            <option value="off" <?php selected( $settings['schedule_frequency'], 'off' ); ?>>
                                <?php esc_html_e( 'Off (manual backups only)', 'adams-backups' ); ?>
                            </option>
                        </select>
                        <p class="description">
                            <?php esc_html_e( 'Backups run overnight using WordPress cron. For best reliability, set up a real server cron to hit wp-cron.php.', 'adams-backups' ); ?>
                        </p>
                    </td>
                </tr>
            </table>

            <?php submit_button(); ?>
        </form>

        <form method="post" style="margin-top: 1em;">
            <?php wp_nonce_field( 'ab_backups_manage_billing', 'ab_backups_manage_billing_nonce' ); ?>
            <?php submit_button( __( 'Manage billing in Adam’s Backups', 'adams-backups' ), 'secondary', 'ab_backups_manage_billing', false ); ?>
        </form>
        <?php
    }
}

