<?php
/**
 * Plugin Name: Adam's Comms
 * Description: Floating WhatsApp / Call / Email bar with a quick contact form, submissions, and click tracking.
 * Version:     0.2.4
 * Author:      Adam / Dori Media
 * Text Domain: adams-comms
 */

if (!defined('ABSPATH')) {
    exit;
}

class Adams_Comms_Plugin {

    const OPTION_BAR_SETTINGS   = 'adams_comms_bar_settings';
    const OPTION_FORM_SETTINGS  = 'adams_comms_form_settings';
    const OPTION_CLICK_COUNTS   = 'adams_comms_click_counts';
    const TABLE_SUBMISSIONS     = 'adams_comms_submissions';

    public function __construct() {
        register_activation_hook(__FILE__, array($this, 'on_activate'));

        add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('wp_footer', array($this, 'render_contact_bar'));

        // Elementor widget support
        add_action('plugins_loaded', array($this, 'init_elementor_integration'));

        add_action('wp_ajax_nopriv_adams_comms_track_click', array($this, 'track_click'));
        add_action('wp_ajax_adams_comms_track_click', array($this, 'track_click'));

        add_action('wp_ajax_nopriv_adams_comms_submit_form', array($this, 'handle_form_submit'));
        add_action('wp_ajax_adams_comms_submit_form', array($this, 'handle_form_submit'));

        add_shortcode('adams_comms_form', array($this, 'render_form_shortcode'));

        if (is_admin()) {
            add_action('admin_menu', array($this, 'register_admin_menu'));
            add_action('admin_init', array($this, 'register_settings'));
        }
    }

    

    /**
     * Initialize Elementor integration: register custom widget if Elementor is active.
     */
    public function init_elementor_integration() {
        if ( ! did_action( 'elementor/loaded' ) ) {
            return;
        }

        // Elementor 3.5+ widgets registration hook.
        add_action( 'elementor/widgets/register', array( $this, 'register_elementor_widget' ) );
    }

    /**
     * Register the Adam's Comms form widget with Elementor.
     *
     * @param \Elementor\Widgets_Manager $widgets_manager
     */
    public function register_elementor_widget( $widgets_manager ) {
        // Make sure Elementor base class is available.
        if ( ! class_exists( '\\Elementor\\Widget_Base' ) ) {
            return;
        }

        // Load widget class file.
        require_once plugin_dir_path( __FILE__ ) . 'includes/class-adams-comms-elementor-form-widget.php';

        if ( class_exists( 'Adams_Comms_Elementor_Form_Widget' ) ) {
            $widgets_manager->register( new \Adams_Comms_Elementor_Form_Widget() );
        }
    }


    public function on_activate() {
        global $wpdb;
        $table_name = $this->get_submissions_table_name();
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS {$table_name} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            name VARCHAR(190) DEFAULT '' NOT NULL,
            email VARCHAR(190) DEFAULT '' NOT NULL,
            phone VARCHAR(190) DEFAULT '' NOT NULL,
            message TEXT NOT NULL,
            page_url TEXT NULL,
            created_at DATETIME NOT NULL,
            PRIMARY KEY  (id)
        ) {$charset_collate};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);

        if (!get_option(self::OPTION_BAR_SETTINGS)) {
            $defaults = array(
                'enabled'          => 1,
                'phone_number'     => '',
                'whatsapp_number'  => '',
                'email_address'    => get_option('admin_email'),
                'whatsapp_message' => "Hi, I'm interested in your services.",
                'show_whatsapp'    => 1,
                'show_call'        => 1,
                'show_email'       => 1,
                'show_form'        => 1,
                'ga4_enabled'      => 0,
                'ga4_event_prefix' => 'adams_comms',
            );
            add_option(self::OPTION_BAR_SETTINGS, $defaults);
        }

        if (!get_option(self::OPTION_FORM_SETTINGS)) {
            $defaults = array(
                'send_to'         => get_option('admin_email'),
                'success_message' => 'Thanks, we\'ve got your message.',
                'redirect_url'    => '',
                'fields'          => array(
                    'name'    => array('enabled' => 1, 'required' => 1, 'label' => 'Name'),
                    'email'   => array('enabled' => 1, 'required' => 1, 'label' => 'Email'),
                    'phone'   => array('enabled' => 1, 'required' => 0, 'label' => 'Phone'),
                    'message' => array('enabled' => 1, 'required' => 1, 'label' => 'Message'),
                ),
            );
            add_option(self::OPTION_FORM_SETTINGS, $defaults);
        }

        if (!get_option(self::OPTION_CLICK_COUNTS)) {
            add_option(self::OPTION_CLICK_COUNTS, array(
                'call'     => 0,
                'whatsapp' => 0,
                'email'    => 0,
                'form'     => 0,
            ));
        }
    }

    public function enqueue_assets() {
        add_shortcode('adams_comms_form', array($this, 'render_form_shortcode'));

        if (is_admin()) {
            return;
        }

        $version = '0.2.1';

        wp_enqueue_style(
            'adams-comms-css',
            plugins_url('assets/css/adams-comms.css', __FILE__),
            array(),
            $version
        );

        wp_enqueue_script(
            'adams-comms-js',
            plugins_url('assets/js/adams-comms.js', __FILE__),
            array('jquery'),
            $version,
            true
        );

        $bar_settings  = $this->get_bar_settings();
        $form_settings = $this->get_form_settings();

        wp_localize_script('adams-comms-js', 'AdamsComms', array(
            'ajaxUrl'        => admin_url('admin-ajax.php'),
            'nonce'          => wp_create_nonce('adams_comms_track'),
            'ga4Enabled'     => !empty($bar_settings['ga4_enabled']) ? 1 : 0,
            'ga4EventPrefix' => !empty($bar_settings['ga4_event_prefix']) ? $bar_settings['ga4_event_prefix'] : 'adams_comms',
            'successMessage' => isset($form_settings['success_message']) ? $form_settings['success_message'] : '',
            'source'         => 'adams-comms',
        ));
    }

    public function render_contact_bar() {
        add_shortcode('adams_comms_form', array($this, 'render_form_shortcode'));

        if (is_admin()) {
            return;
        }

        $bar  = $this->get_bar_settings();
        $form = $this->get_form_settings();

        if (empty($bar['enabled'])) {
            return;
        }

        $phone_href    = $this->format_phone_tel(isset($bar['phone_number']) ? $bar['phone_number'] : '');
        $whatsapp_link = $this->get_whatsapp_link(
            isset($bar['whatsapp_number']) ? $bar['whatsapp_number'] : '',
            isset($bar['whatsapp_message']) ? $bar['whatsapp_message'] : ''
        );
        $email_address = isset($bar['email_address']) ? $bar['email_address'] : get_option('admin_email');

        ?>
        <div id="adams-comms-bar" aria-label="<?php esc_attr_e('Contact options', 'adams-comms'); ?>" role="complementary">
            <?php if (!empty($bar['show_call']) && $phone_href) : ?>
                <button class="ac-btn ac-btn-call" data-ac-type="call"
                        aria-label="<?php esc_attr_e('Call us', 'adams-comms'); ?>"
                        onclick="window.location.href='<?php echo esc_js($phone_href); ?>'">
                    <span class="ac-icon">📞</span>
                    <span class="ac-label"><?php esc_html_e('Call', 'adams-comms'); ?></span>
                </button>
            <?php endif; ?>

            <?php if (!empty($bar['show_whatsapp']) && $whatsapp_link) : ?>
                <button class="ac-btn ac-btn-whatsapp" data-ac-type="whatsapp"
                        aria-label="<?php esc_attr_e('Chat on WhatsApp', 'adams-comms'); ?>"
                        onclick="window.location.href='<?php echo esc_js($whatsapp_link); ?>'">
                    <span class="ac-icon">🟢</span>
                    <span class="ac-label"><?php esc_html_e('WhatsApp', 'adams-comms'); ?></span>
                </button>
            <?php endif; ?>

            <?php if (!empty($bar['show_email']) && !empty($email_address)) : ?>
                <button class="ac-btn ac-btn-email" data-ac-type="email"
                        aria-label="<?php esc_attr_e('Email us', 'adams-comms'); ?>"
                        onclick="window.location.href='mailto:<?php echo esc_js($email_address); ?>'">
                    <span class="ac-icon">✉</span>
                    <span class="ac-label"><?php esc_html_e('Email', 'adams-comms'); ?></span>
                </button>
            <?php endif; ?>

            <?php if (!empty($bar['show_form'])) : ?>
                <button class="ac-btn ac-btn-message" data-ac-type="form"
                        aria-label="<?php esc_attr_e('Send us a message', 'adams-comms'); ?>"
                        data-ac-toggle="form">
                    <span class="ac-icon">📝</span>
                    <span class="ac-label"><?php esc_html_e('Message', 'adams-comms'); ?></span>
                </button>
            <?php endif; ?>
        </div>
        <?php
        $fields = isset($form['fields']) && is_array($form['fields']) ? $form['fields'] : array();
        ?>
        <div id="adams-comms-overlay" class="ac-overlay" aria-hidden="true">
            <div class="ac-overlay-inner" role="dialog" aria-modal="true" aria-labelledby="adams-comms-form-title">
                <button class="ac-overlay-close" type="button" aria-label="<?php esc_attr_e('Close form', 'adams-comms'); ?>" data-ac-close>
                    ×
                </button>
                <h2 id="adams-comms-form-title" class="ac-overlay-title">
                    <?php esc_html_e('Message us', 'adams-comms'); ?>
                </h2>
                <p class="ac-overlay-subtitle">
                    <?php esc_html_e('Fill in the form below and we’ll get back to you.', 'adams-comms'); ?>
                </p>

                <form class="ac-form ac-form-overlay" data-ac-form="overlay">
                    <?php if (!empty($fields['name']['enabled'])) : ?>
                        <div class="ac-field">
                            <label for="ac-name">
                                <?php echo esc_html($fields['name']['label']); ?><?php echo !empty($fields['name']['required']) ? ' *' : ''; ?>
                            </label>
                            <input type="text" id="ac-name" name="name"
                                   <?php echo !empty($fields['name']['required']) ? 'required' : ''; ?>>
                        </div>
                    <?php endif; ?>

                    <?php if (!empty($fields['email']['enabled'])) : ?>
                        <div class="ac-field">
                            <label for="ac-email">
                                <?php echo esc_html($fields['email']['label']); ?><?php echo !empty($fields['email']['required']) ? ' *' : ''; ?>
                            </label>
                            <input type="email" id="ac-email" name="email"
                                   <?php echo !empty($fields['email']['required']) ? 'required' : ''; ?>>
                        </div>
                    <?php endif; ?>

                    <?php if (!empty($fields['phone']['enabled'])) : ?>
                        <div class="ac-field">
                            <label for="ac-phone">
                                <?php echo esc_html($fields['phone']['label']); ?><?php echo !empty($fields['phone']['required']) ? ' *' : ''; ?>
                            </label>
                            <input type="tel" id="ac-phone" name="phone"
                                   <?php echo !empty($fields['phone']['required']) ? 'required' : ''; ?>>
                        </div>
                    <?php endif; ?>

                    <?php if (!empty($fields['message']['enabled'])) : ?>
                        <div class="ac-field">
                            <label for="ac-message">
                                <?php echo esc_html($fields['message']['label']); ?><?php echo !empty($fields['message']['required']) ? ' *' : ''; ?>
                            </label>
                            <textarea id="ac-message" name="message" rows="4"
                                      <?php echo !empty($fields['message']['required']) ? 'required' : ''; ?>></textarea>
                        </div>
                    <?php endif; ?>

                    <button type="submit" class="ac-submit">
                        <?php esc_html_e('Send message', 'adams-comms'); ?>
                    </button>

                    <p class="ac-success" style="display:none;"></p>
                    <p class="ac-error" style="display:none;"><?php esc_html_e('Something went wrong. Please try again.', 'adams-comms'); ?></p>
                </form>
            </div>
        </div>
        <?php
    }

    
    /**
     * Shortcode: [adams_comms_form] – render the same quick contact form for use in Elementor or pages.
     */
    public function render_form_shortcode($atts = array(), $content = null) {
        if (is_admin() && !wp_doing_ajax()) {
            return '';
        }

        $form   = $this->get_form_settings();
        $fields = isset($form['fields']) && is_array($form['fields']) ? $form['fields'] : array();

        ob_start();
        ?>
        <div class="adams-comms-form-wrapper">
            <form class="ac-form ac-form-shortcode" data-ac-form="shortcode">
                <?php if (!empty($fields['name']['enabled'])) : ?>
                    <div class="ac-field">
                        <label for="ac-name-short">
                            <?php echo esc_html($fields['name']['label']); ?><?php echo !empty($fields['name']['required']) ? ' *' : ''; ?>
                        </label>
                        <input type="text" id="ac-name-short" name="name"
                               <?php echo !empty($fields['name']['required']) ? 'required' : ''; ?>>
                    </div>
                <?php endif; ?>

                <?php if (!empty($fields['email']['enabled'])) : ?>
                    <div class="ac-field">
                        <label for="ac-email-short">
                            <?php echo esc_html($fields['email']['label']); ?><?php echo !empty($fields['email']['required']) ? ' *' : ''; ?>
                        </label>
                        <input type="email" id="ac-email-short" name="email"
                               <?php echo !empty($fields['email']['required']) ? 'required' : ''; ?>>
                    </div>
                <?php endif; ?>

                <?php if (!empty($fields['phone']['enabled'])) : ?>
                    <div class="ac-field">
                        <label for="ac-phone-short">
                            <?php echo esc_html($fields['phone']['label']); ?><?php echo !empty($fields['phone']['required']) ? ' *' : ''; ?>
                        </label>
                        <input type="tel" id="ac-phone-short" name="phone"
                               <?php echo !empty($fields['phone']['required']) ? 'required' : ''; ?>>
                    </div>
                <?php endif; ?>

                <?php if (!empty($fields['message']['enabled'])) : ?>
                    <div class="ac-field">
                        <label for="ac-message-short">
                            <?php echo esc_html($fields['message']['label']); ?><?php echo !empty($fields['message']['required']) ? ' *' : ''; ?>
                        </label>
                        <textarea id="ac-message-short" name="message" rows="4"
                                  <?php echo !empty($fields['message']['required']) ? 'required' : ''; ?>></textarea>
                    </div>
                <?php endif; ?>

                <button type="submit" class="ac-submit">
                    <?php esc_html_e('Send message', 'adams-comms'); ?>
                </button>

                <p class="ac-success" style="display:none;"></p>
                <p class="ac-error" style="display:none;"><?php esc_html_e('Something went wrong. Please try again.', 'adams-comms'); ?></p>
            </form>
        </div>
        <?php
        return ob_get_clean();
    }

public function track_click() {
        check_ajax_referer('adams_comms_track', 'nonce');

        $type = isset($_POST['type']) ? sanitize_key($_POST['type']) : '';
        $allowed = array('call', 'whatsapp', 'email', 'form');

        if (!in_array($type, $allowed, true)) {
            wp_send_json_error(array('message' => 'Invalid type'));
        }

        $counts = get_option(self::OPTION_CLICK_COUNTS, array(
            'call'     => 0,
            'whatsapp' => 0,
            'email'    => 0,
            'form'     => 0,
        ));

        if (!isset($counts[$type])) {
            $counts[$type] = 0;
        }
        $counts[$type]++;

        update_option(self::OPTION_CLICK_COUNTS, $counts);

        wp_send_json_success();
    }

    public function handle_form_submit() {
        check_ajax_referer('adams_comms_track', 'nonce');

        $form = $this->get_form_settings();

        $name     = isset($_POST['name']) ? sanitize_text_field(wp_unslash($_POST['name'])) : '';
        $email    = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        $phone    = isset($_POST['phone']) ? sanitize_text_field(wp_unslash($_POST['phone'])) : '';
        $message  = isset($_POST['message']) ? wp_kses_post(wp_unslash($_POST['message'])) : '';
        $page_url = isset($_POST['page_url']) ? esc_url_raw(wp_unslash($_POST['page_url'])) : '';

        $fields = isset($form['fields']) && is_array($form['fields']) ? $form['fields'] : array();

        if (!empty($fields['name']['enabled']) && !empty($fields['name']['required']) && $name === '') {
            wp_send_json_error(array('message' => __('Name is required.', 'adams-comms')));
        }
        if (!empty($fields['email']['enabled']) && !empty($fields['email']['required']) && $email === '') {
            wp_send_json_error(array('message' => __('Email is required.', 'adams-comms')));
        }
        if (!empty($fields['message']['enabled']) && !empty($fields['message']['required']) && $message === '') {
            wp_send_json_error(array('message' => __('Message is required.', 'adams-comms')));
        }

        global $wpdb;
        $table_name = $this->get_submissions_table_name();
        $wpdb->insert(
            $table_name,
            array(
                'name'       => $name,
                'email'      => $email,
                'phone'      => $phone,
                'message'    => $message,
                'page_url'   => $page_url,
                'created_at' => current_time('mysql'),
            ),
            array('%s', '%s', '%s', '%s', '%s', '%s')
        );

        $to      = !empty($form['send_to']) ? $form['send_to'] : get_option('admin_email');
        $subject = sprintf(__('New enquiry from %s', 'adams-comms'), get_bloginfo('name'));
        $body    = "New Adam's Comms submission:

";
        $body   .= "Name: {$name}
";
        $body   .= "Email: {$email}
";
        $body   .= "Phone: {$phone}
";
        $body   .= "Message:
{$message}

";
        if ($page_url) {
            $body .= "Page URL: {$page_url}
";
        }

        $headers = array('Content-Type: text/plain; charset=UTF-8');

        wp_mail($to, $subject, $body, $headers);

        $counts = get_option(self::OPTION_CLICK_COUNTS, array(
            'call'     => 0,
            'whatsapp' => 0,
            'email'    => 0,
            'form'     => 0,
        ));
        if (!isset($counts['form'])) {
            $counts['form'] = 0;
        }
        $counts['form']++;
        update_option(self::OPTION_CLICK_COUNTS, $counts);

        $success_message = isset($form['success_message']) ? $form['success_message'] : __('Thanks, we\'ve got your message.', 'adams-comms');

        wp_send_json_success(array(
            'message'      => $success_message,
            'redirect_url' => isset($form['redirect_url']) ? $form['redirect_url'] : '',
        ));
    }

    public function register_admin_menu() {
        add_menu_page(
            __('Adam\'s Comms', 'adams-comms'),
            __('Adam\'s Comms', 'adams-comms'),
            'manage_options',
            'adams-comms',
            array($this, 'render_dashboard_page'),
            'dashicons-megaphone',
            58
        );

        add_submenu_page(
            'adams-comms',
            __('Dashboard', 'adams-comms'),
            __('Dashboard', 'adams-comms'),
            'manage_options',
            'adams-comms',
            array($this, 'render_dashboard_page')
        );

        add_submenu_page(
            'adams-comms',
            __('Contact Bar', 'adams-comms'),
            __('Contact Bar', 'adams-comms'),
            'manage_options',
            'adams-comms-bar',
            array($this, 'render_bar_settings_page')
        );

        add_submenu_page(
            'adams-comms',
            __('Form', 'adams-comms'),
            __('Form', 'adams-comms'),
            'manage_options',
            'adams-comms-form',
            array($this, 'render_form_settings_page')
        );
    }

    public function register_settings() {
        register_setting('adams_comms_bar_group', self::OPTION_BAR_SETTINGS);
        register_setting('adams_comms_form_group', self::OPTION_FORM_SETTINGS);
    }

    public function render_dashboard_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $counts = get_option(self::OPTION_CLICK_COUNTS, array(
            'call'     => 0,
            'whatsapp' => 0,
            'email'    => 0,
            'form'     => 0,
        ));

        global $wpdb;
        $table_name = $this->get_submissions_table_name();
        $total_subs = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table_name}");

        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Adam\'s Comms Dashboard', 'adams-comms'); ?></h1>
            <p><?php esc_html_e('Overview of how visitors are contacting you via Adam\'s Comms.', 'adams-comms'); ?></p>

            <div class="ac-admin-cards">
                <div class="ac-admin-card">
                    <h2><?php esc_html_e('WhatsApp clicks', 'adams-comms'); ?></h2>
                    <p class="ac-admin-number"><?php echo (int) $counts['whatsapp']; ?></p>
                </div>
                <div class="ac-admin-card">
                    <h2><?php esc_html_e('Call clicks', 'adams-comms'); ?></h2>
                    <p class="ac-admin-number"><?php echo (int) $counts['call']; ?></p>
                </div>
                <div class="ac-admin-card">
                    <h2><?php esc_html_e('Email clicks', 'adams-comms'); ?></h2>
                    <p class="ac-admin-number"><?php echo (int) $counts['email']; ?></p>
                </div>
                <div class="ac-admin-card">
                    <h2><?php esc_html_e('Form submissions', 'adams-comms'); ?></h2>
                    <p class="ac-admin-number"><?php echo (int) $total_subs; ?></p>
                </div>
            </div>

            <p style="margin-top:24px;">
                <?php esc_html_e('More detailed reporting and per-page breakdowns can be added in a future version.', 'adams-comms'); ?>
            </p>
        </div>
        <style>
            .ac-admin-cards {
                display:flex;
                flex-wrap:wrap;
                gap:16px;
                margin-top:16px;
            }
            .ac-admin-card {
                flex:1 1 200px;
                background:#fff;
                border:1px solid #ddd;
                border-radius:8px;
                padding:12px 16px;
            }
            .ac-admin-number {
                font-size:28px;
                font-weight:700;
                margin:8px 0 0;
            }
        </style>
        <?php
    }

    public function render_bar_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $options = $this->get_bar_settings();
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Contact Bar Settings', 'adams-comms'); ?></h1>
            <form method="post" action="options.php">
                <?php settings_fields('adams_comms_bar_group'); ?>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e('Enable contact bar', 'adams-comms'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPTION_BAR_SETTINGS); ?>[enabled]" value="1" <?php checked(!empty($options['enabled'])); ?>>
                                <?php esc_html_e('Show the contact bar on the front-end', 'adams-comms'); ?>
                            </label>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e('Phone number', 'adams-comms'); ?></th>
                        <td>
                            <input type="text" class="regular-text" name="<?php echo esc_attr(self::OPTION_BAR_SETTINGS); ?>[phone_number]" value="<?php echo esc_attr($options['phone_number']); ?>">
                            <p class="description"><?php esc_html_e('Local number, e.g. 021 123 4567.', 'adams-comms'); ?></p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e('WhatsApp number', 'adams-comms'); ?></th>
                        <td>
                            <input type="text" class="regular-text" name="<?php echo esc_attr(self::OPTION_BAR_SETTINGS); ?>[whatsapp_number]" value="<?php echo esc_attr($options['whatsapp_number']); ?>">
                            <p class="description"><?php esc_html_e('We will convert this to international format for wa.me links.', 'adams-comms'); ?></p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e('Default WhatsApp message', 'adams-comms'); ?></th>
                        <td>
                            <textarea class="large-text" rows="3" name="<?php echo esc_attr(self::OPTION_BAR_SETTINGS); ?>[whatsapp_message]"><?php echo esc_textarea($options['whatsapp_message']); ?></textarea>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e('Email address', 'adams-comms'); ?></th>
                        <td>
                            <input type="email" class="regular-text" name="<?php echo esc_attr(self::OPTION_BAR_SETTINGS); ?>[email_address]" value="<?php echo esc_attr($options['email_address']); ?>">
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e('Show buttons', 'adams-comms'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPTION_BAR_SETTINGS); ?>[show_call]" value="1" <?php checked(!empty($options['show_call'])); ?>>
                                <?php esc_html_e('Show Call button', 'adams-comms'); ?>
                            </label>
                            <br>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPTION_BAR_SETTINGS); ?>[show_whatsapp]" value="1" <?php checked(!empty($options['show_whatsapp'])); ?>>
                                <?php esc_html_e('Show WhatsApp button', 'adams-comms'); ?>
                            </label>
                            <br>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPTION_BAR_SETTINGS); ?>[show_email]" value="1" <?php checked(!empty($options['show_email'])); ?>>
                                <?php esc_html_e('Show Email button', 'adams-comms'); ?>
                            </label>
                            <br>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPTION_BAR_SETTINGS); ?>[show_form]" value="1" <?php checked(!empty($options['show_form'])); ?>>
                                <?php esc_html_e('Show Message form button', 'adams-comms'); ?>
                            </label>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e('Google Analytics 4', 'adams-comms'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPTION_BAR_SETTINGS); ?>[ga4_enabled]" value="1" <?php checked(!empty($options['ga4_enabled'])); ?>>
                                <?php esc_html_e('Send events to GA4 (gtag.js)', 'adams-comms'); ?>
                            </label>
                            <p class="description">
                                <?php esc_html_e('Events will be fired using your existing gtag.js setup.', 'adams-comms'); ?>
                            </p>
                            <p>
                                <label>
                                    <?php esc_html_e('Event name prefix', 'adams-comms'); ?>:
                                    <input type="text" name="<?php echo esc_attr(self::OPTION_BAR_SETTINGS); ?>[ga4_event_prefix]" value="<?php echo esc_attr($options['ga4_event_prefix']); ?>">
                                </label>
                                <span class="description"><?php esc_html_e('Default is "adams_comms". Click events use this prefix plus "_click".', 'adams-comms'); ?></span>
                            </p>
                        </td>
                    </tr>
                </table>

                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    public function render_form_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $form   = $this->get_form_settings();
        $fields = isset($form['fields']) && is_array($form['fields']) ? $form['fields'] : array();
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Form Settings', 'adams-comms'); ?></h1>
            <p><?php esc_html_e('Configure the quick contact form that appears when visitors click "Message".', 'adams-comms'); ?></p>

            <form method="post" action="options.php">
                <?php settings_fields('adams_comms_form_group'); ?>

                <h2 class="title"><?php esc_html_e('General', 'adams-comms'); ?></h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e('Send submissions to', 'adams-comms'); ?></th>
                        <td>
                            <input type="email" class="regular-text" name="<?php echo esc_attr(self::OPTION_FORM_SETTINGS); ?>[send_to]" value="<?php echo esc_attr($form['send_to']); ?>">
                            <p class="description"><?php esc_html_e('Email address to receive notifications.', 'adams-comms'); ?></p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e('Success message', 'adams-comms'); ?></th>
                        <td>
                            <textarea class="large-text" rows="3" name="<?php echo esc_attr(self::OPTION_FORM_SETTINGS); ?>[success_message]"><?php echo esc_textarea($form['success_message']); ?></textarea>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e('Redirect URL (optional)', 'adams-comms'); ?></th>
                        <td>
                            <input type="url" class="regular-text" name="<?php echo esc_attr(self::OPTION_FORM_SETTINGS); ?>[redirect_url]" value="<?php echo esc_attr($form['redirect_url']); ?>">
                            <p class="description"><?php esc_html_e('If set, visitors will be redirected here after a successful submission.', 'adams-comms'); ?></p>
                        </td>
                    </tr>
                </table>

                <h2 class="title"><?php esc_html_e('Fields', 'adams-comms'); ?></h2>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Field', 'adams-comms'); ?></th>
                            <th><?php esc_html_e('Enabled', 'adams-comms'); ?></th>
                            <th><?php esc_html_e('Required', 'adams-comms'); ?></th>
                            <th><?php esc_html_e('Label', 'adams-comms'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $field_keys = array('name', 'email', 'phone', 'message');
                        foreach ($field_keys as $key) :
                            $field = isset($fields[$key]) ? $fields[$key] : array('enabled' => 1, 'required' => 0, 'label' => ucfirst($key));
                            ?>
                            <tr>
                                <td><strong><?php echo esc_html(ucfirst($key)); ?></strong></td>
                                <td>
                                    <input type="checkbox" name="<?php echo esc_attr(self::OPTION_FORM_SETTINGS); ?>[fields][<?php echo esc_attr($key); ?>][enabled]" value="1" <?php checked(!empty($field['enabled'])); ?>>
                                </td>
                                <td>
                                    <input type="checkbox" name="<?php echo esc_attr(self::OPTION_FORM_SETTINGS); ?>[fields][<?php echo esc_attr($key); ?>][required]" value="1" <?php checked(!empty($field['required'])); ?>>
                                </td>
                                <td>
                                    <input type="text" class="regular-text" name="<?php echo esc_attr(self::OPTION_FORM_SETTINGS); ?>[fields][<?php echo esc_attr($key); ?>][label]" value="<?php echo esc_attr($field['label']); ?>">
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <?php submit_button(); ?>
            </form>

            <h2 style="margin-top:30px;"><?php esc_html_e('Recent submissions', 'adams-comms'); ?></h2>
            <?php
            global $wpdb;
            $table_name = $this->get_submissions_table_name();
            $rows = $wpdb->get_results("SELECT * FROM {$table_name} ORDER BY created_at DESC LIMIT 20");
            if ($rows) :
                ?>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Date', 'adams-comms'); ?></th>
                            <th><?php esc_html_e('Name', 'adams-comms'); ?></th>
                            <th><?php esc_html_e('Email', 'adams-comms'); ?></th>
                            <th><?php esc_html_e('Phone', 'adams-comms'); ?></th>
                            <th><?php esc_html_e('Message', 'adams-comms'); ?></th>
                            <th><?php esc_html_e('Page', 'adams-comms'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($rows as $row) : ?>
                            <tr>
                                <td><?php echo esc_html($row->created_at); ?></td>
                                <td><?php echo esc_html($row->name); ?></td>
                                <td><?php echo esc_html($row->email); ?></td>
                                <td><?php echo esc_html($row->phone); ?></td>
                                <td><?php echo esc_html(wp_html_excerpt($row->message, 80, '...')); ?></td>
                                <td>
                                    <?php if (!empty($row->page_url)) : ?>
                                        <a href="<?php echo esc_url($row->page_url); ?>" target="_blank" rel="noopener noreferrer">
                                            <?php echo esc_html(wp_html_excerpt($row->page_url, 40, '...')); ?>
                                        </a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else : ?>
                <p><?php esc_html_e('No submissions yet.', 'adams-comms'); ?></p>
            <?php endif; ?>
        </div>
        <?php
    }

    private function get_bar_settings() {
        $defaults = array(
            'enabled'          => 1,
            'phone_number'     => '',
            'whatsapp_number'  => '',
            'whatsapp_message' => '',
            'email_address'    => get_option('admin_email'),
            'show_whatsapp'    => 1,
            'show_call'        => 1,
            'show_email'       => 1,
            'show_form'        => 1,
            'ga4_enabled'      => 0,
            'ga4_event_prefix' => 'adams_comms',
        );
        $options = get_option(self::OPTION_BAR_SETTINGS, array());
        return wp_parse_args($options, $defaults);
    }

    private function get_form_settings() {
        $defaults = array(
            'send_to'         => get_option('admin_email'),
            'success_message' => 'Thanks, we\'ve got your message.',
            'redirect_url'    => '',
            'fields'          => array(
                'name'    => array('enabled' => 1, 'required' => 1, 'label' => 'Name'),
                'email'   => array('enabled' => 1, 'required' => 1, 'label' => 'Email'),
                'phone'   => array('enabled' => 1, 'required' => 0, 'label' => 'Phone'),
                'message' => array('enabled' => 1, 'required' => 1, 'label' => 'Message'),
            ),
        );
        $options = get_option(self::OPTION_FORM_SETTINGS, array());
        return wp_parse_args($options, $defaults);
    }

    private function get_submissions_table_name() {
        global $wpdb;
        return $wpdb->prefix . self::TABLE_SUBMISSIONS;
    }

    private function format_phone_tel($raw) {
        $digits = preg_replace('/\D+/', '', $raw);
        if ($digits === '') {
            return '';
        }
        return 'tel:' . $digits;
    }

    private function get_whatsapp_link($number_raw, $message = '') {
        $digits = preg_replace('/\D+/', '', $number_raw);
        if ($digits === '') {
            return '';
        }

        if (strpos($digits, '0') === 0 && strlen($digits) > 3) {
            $digits = '64' . substr($digits, 1);
        }

        $base = 'https://wa.me/' . $digits;

        if (!empty($message)) {
            $base .= '?text=' . rawurlencode($message);
        }

        return $base;
    }
}

new Adams_Comms_Plugin();
