jQuery(function($) {
    var $bar = $('#adams-comms-bar');
    var $overlay = $('#adams-comms-overlay');

    function openOverlay() {
        $overlay.addClass('ac-overlay--open').attr('aria-hidden', 'false');
    }

    function closeOverlay() {
        $overlay.removeClass('ac-overlay--open').attr('aria-hidden', 'true');
    }

    // Toggle overlay when Message button is clicked
    $bar.on('click', '[data-ac-toggle="form"]', function(e) {
        e.preventDefault();
        trackClick('form', { ui: 'bar' });
        openOverlay();
    });

    // Close overlay buttons
    $overlay.on('click', '[data-ac-close]', function(e) {
        e.preventDefault();
        closeOverlay();
    });

    // Click outside inner panel closes overlay
    $overlay.on('click', function(e) {
        if (e.target === this) {
            closeOverlay();
        }
    });

    // Track clicks on other buttons
    $bar.on('click', '[data-ac-type]', function() {
        var type = $(this).data('ac-type');
        if (type && type !== 'form') {
            trackClick(type, { ui: 'bar' });
        }
    });

    // Core tracking helper, pushes to WP, GA4, and Adam's Analytics
    function trackClick(type, extraMeta) {
        extraMeta = extraMeta || {};

        if (!window.AdamsComms || !AdamsComms.ajaxUrl) {
            return;
        }

        var page = window.location.href;
        var payload = {
            action: 'adams_comms_track_click',
            nonce: AdamsComms.nonce,
            type: type,
            url: page
        };

        $.post(AdamsComms.ajaxUrl, payload);

        // GA4
        if (AdamsComms.ga4Enabled && typeof window.gtag === 'function') {
            var eventName = (AdamsComms.ga4EventPrefix || 'adams_comms') + '_click';
            window.gtag('event', eventName, {
                event_category: 'contact',
                event_label: type,
                contact_type: type,
                page_location: page,
                source: AdamsComms.source || 'adams-comms'
            });
        }

        // Adam's Analytics – standardised schema
        if (window.AdamsAnalytics && typeof window.AdamsAnalytics.trackEvent === 'function') {
            var meta = {
                channel: type,
                page: page,
                source: AdamsComms.source || 'adams-comms',
                ui: extraMeta.ui || 'bar'
            };
            window.AdamsAnalytics.trackEvent('adams_comms_click', meta);
        }
    }

    // Form submission for both overlay and shortcode forms
    $(document).on('submit', '.ac-form', function(e) {
        e.preventDefault();

        var $form = $(this);
        var $success = $form.find('.ac-success');
        var $error = $form.find('.ac-error');

        $success.hide();
        $error.hide();

        var formEl = $form[0];
        if (!formEl.checkValidity()) {
            formEl.reportValidity();
            return;
        }

        var data = {
            action: 'adams_comms_submit_form',
            nonce: AdamsComms.nonce,
            page_url: window.location.href
        };

        $form.serializeArray().forEach(function(item) {
            data[item.name] = item.value;
        });

        $.post(AdamsComms.ajaxUrl, data)
            .done(function(resp) {
                if (resp && resp.success) {
                    var msg = (resp.data && resp.data.message) ? resp.data.message : (AdamsComms.successMessage || 'Thanks, we\'ve got your message.');
                    $success.text(msg).show();
                    $error.hide();

                    var inOverlay = $form.closest('#adams-comms-overlay').length > 0;
                    var uiContext = inOverlay ? 'overlay' : 'shortcode';

                    // Track as both a click + dedicated form_submit event in Adam's Analytics
                    trackClick('form', { ui: uiContext });

                    if (window.AdamsAnalytics && typeof window.AdamsAnalytics.trackEvent === 'function') {
                        window.AdamsAnalytics.trackEvent('adams_comms_form_submit', {
                            page: window.location.href,
                            source: AdamsComms.source || 'adams-comms',
                            ui: uiContext
                        });
                    }

                    // Clear fields
                    $form.find('input[type="text"], input[type="email"], input[type="tel"], textarea').val('');

                    if (resp.data && resp.data.redirect_url) {
                        window.location.href = resp.data.redirect_url;
                        return;
                    }

                    if (inOverlay) {
                        setTimeout(function() {
                            closeOverlay();
                            $success.hide();
                        }, 2000);
                    }
                } else {
                    $error.show();
                }
            })
            .fail(function() {
                $error.show();
            });
    });
});
