<?php
/**
 * Plugin Name: Adam's Dashboard Cleaner
 * Description: Makes WordPress feel simple for clients – clean dashboard, friendly labels, trimmed admin menu, and an Ask Adam support form.
 * Author: Dori Media / Adam Crouchley
 * Version: 1.2.1
 * License: GPL2+
 * Text Domain: adams-dashboard-cleaner
 */

if (!defined('ABSPATH')) {
    exit;
}

class ADC_Dashboard_Cleaner {

    const OPTION_KEY = 'adc_dashboard_cleaner_options';

    private static $instance = null;

    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        // Activation defaults
        register_activation_hook(__FILE__, array($this, 'activate'));

        // Admin UI
        add_action('admin_menu', array($this, 'register_settings_page'));
        add_action('admin_init', array($this, 'register_settings'));

        // Dashboard widgets
        add_action('wp_dashboard_setup', array($this, 'clean_dashboard_widgets'));
        add_action('wp_dashboard_setup', array($this, 'add_start_here_widget'));
        add_action('wp_dashboard_setup', array($this, 'add_ask_adam_widget'));

        // Ask Adam form handler + notices
        add_action('admin_init', array($this, 'handle_ask_adam_form'));
        add_action('admin_notices', array($this, 'render_ask_adam_notice'));

        // Menu tweaks for non-admins
        add_action('admin_menu', array($this, 'filter_admin_menu'), 999);

        // Rename menu labels
        add_action('admin_menu', array($this, 'rename_admin_menus'), 20);

        // Styling
        add_action('admin_head', array($this, 'output_admin_css'));
    }

    /* ========================================================
     * Options
     * ===================================================== */

    public function activate() {
        $defaults = array(
            'hide_default_dashboard_widgets' => 1,
            'add_start_here_widget'         => 1,

            // Hide menu items for roles that DO NOT have manage_options
            'hide_comments'                 => 1,
            'hide_tools'                    => 1,
            'hide_settings'                 => 1,
            'hide_plugins'                  => 0,
            'hide_appearance'               => 0,
            'hide_users'                    => 0,
            'hide_dashboard'                => 0,

            // Rename menu labels globally
            'rename_posts_to_blog'          => 1,
            'rename_media_to_library'       => 1,
            'rename_pages_to_pages'         => 0,

            // Ask Adam email
            'ask_adam_email'                => 'apcrouchley@gmail.com',
        );

        $current = get_option(self::OPTION_KEY, array());
        update_option(self::OPTION_KEY, wp_parse_args($current, $defaults));
    }

    private function get_options() {
        $defaults = array(
            'hide_default_dashboard_widgets' => 1,
            'add_start_here_widget'         => 1,
            'hide_comments'                 => 1,
            'hide_tools'                    => 1,
            'hide_settings'                 => 1,
            'hide_plugins'                  => 0,
            'hide_appearance'               => 0,
            'hide_users'                    => 0,
            'hide_dashboard'                => 0,
            'rename_posts_to_blog'          => 1,
            'rename_media_to_library'       => 1,
            'rename_pages_to_pages'         => 0,
            'ask_adam_email'                => 'apcrouchley@gmail.com',
        );
        $opts = get_option(self::OPTION_KEY, array());
        return wp_parse_args($opts, $defaults);
    }

    /* ========================================================
     * Dashboard widgets
     * ===================================================== */

    public function clean_dashboard_widgets() {
        $opts = $this->get_options();
        if (empty($opts['hide_default_dashboard_widgets'])) {
            return;
        }

        // Core widgets
        remove_meta_box('dashboard_quick_press', 'dashboard', 'side');
        remove_meta_box('dashboard_primary', 'dashboard', 'side');
        remove_meta_box('dashboard_secondary', 'dashboard', 'side');
        remove_meta_box('dashboard_right_now', 'dashboard', 'normal');
        remove_meta_box('dashboard_activity', 'dashboard', 'normal');

        // Common plugin widgets
        remove_meta_box('e-dashboard-overview', 'dashboard', 'normal'); // Elementor
        remove_meta_box('rg_forms_dashboard', 'dashboard', 'normal');   // Gravity Forms
    }

    public function add_start_here_widget() {
        $opts = $this->get_options();
        if (empty($opts['add_start_here_widget'])) {
            return;
        }

        wp_add_dashboard_widget(
            'adc_start_here_widget',
            "Adam's Start Here",
            array($this, 'render_start_here_widget')
        );
    }

    public function render_start_here_widget() {
        $current_user = wp_get_current_user();
        $is_admin     = current_user_can('manage_options');
        $site_name    = get_bloginfo('name');
        ?>
        <div class="adc-widget">
            <div class="adc-widget-header">
                <h3>Welcome to <?php echo esc_html($site_name); ?></h3>
                <p>Here are the main things you’ll do on this site. Start with the first card and work your way down.</p>
            </div>

            <div class="adc-widget-grid">
                <div class="adc-card">
                    <h4>📝 Write a new post</h4>
                    <p>Share a blog article or news update. Keep it simple and write like you’re talking to a real person.</p>
                    <a class="button button-primary" href="<?php echo esc_url(admin_url('post-new.php')); ?>">Add new post</a>
                    <p class="adc-small">Tip: on the post edit screen, look for “Adam’s SEO Manager” for plain-English SEO advice.</p>
                </div>

                <div class="adc-card">
                    <h4>🏠 Edit your homepage</h4>
                    <p>Update the main content people see first when they visit your site.</p>
                    <?php
                    $front_id = (int) get_option('page_on_front');
                    if ($front_id) {
                        $edit_link = get_edit_post_link($front_id, '');
                        if (function_exists('elementor_load_plugin_textdomain')) {
                            $el_link = add_query_arg(
                                array('post' => $front_id, 'action' => 'elementor'),
                                admin_url('post.php')
                            );
                            echo '<a class="button button-secondary" href="' . esc_url($el_link) . '">Edit with Elementor</a> ';
                        }
                        echo '<a class="button" href="' . esc_url($edit_link); ?>">&nbsp;Edit page</a>
                        <?php
                    } else {
                        echo '<p class="adc-small">This site doesn’t use a static homepage yet. You can choose one under <strong>Settings → Reading</strong>.</p>';
                    }
                    ?>
                </div>

                <div class="adc-card">
                    <h4>📨 Check enquiries</h4>
                    <p>Look for messages and orders from your visitors.</p>
                    <ul class="adc-links-list">
                        <?php if (class_exists('WPCF7_ContactForm')) : ?>
                            <li><a href="<?php echo esc_url(admin_url('admin.php?page=wpcf7')); ?>">Contact Form 7 forms</a></li>
                        <?php endif; ?>
                        <?php if (class_exists('GFForms')) : ?>
                            <li><a href="<?php echo esc_url(admin_url('admin.php?page=gf_entries')); ?>">Gravity Forms entries</a></li>
                        <?php endif; ?>
                        <?php if (class_exists('WooCommerce')) : ?>
                            <li><a href="<?php echo esc_url(admin_url('edit.php?post_type=shop_order')); ?>">WooCommerce orders</a></li>
                        <?php endif; ?>
                        <li><a href="<?php echo esc_url(admin_url('edit-comments.php')); ?>">Comments</a></li>
                    </ul>
                    <p class="adc-small">Not sure where a form sends emails? Do a quick test message and check your inbox.</p>
                </div>

                <div class="adc-card adc-card-help">
                    <h4>🧭 General tips</h4>
                    <p>Worried you’ll break something? That’s normal. A few simple habits make life easier:</p>
                    <ul class="adc-links-list">
                        <li>Update plugins and themes regularly (ask your web person how often).</li>
                        <li>Make small changes and check the site after each one.</li>
                        <li>Keep your login details safe and unique.</li>
                    </ul>
                    <?php if ($is_admin) : ?>
                        <p class="adc-small">As an admin, you can tweak what your team sees under <strong>Settings → Adam’s Dashboard Cleaner</strong>.</p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }

    public function add_ask_adam_widget() {
        wp_add_dashboard_widget(
            'adc_ask_adam_widget',
            "Ask Adam (Support)",
            array($this, 'render_ask_adam_widget')
        );
    }

    public function render_ask_adam_widget() {
        $current_user = wp_get_current_user();
        $opts         = $this->get_options();
        $ask_email    = !empty($opts['ask_adam_email']) ? $opts['ask_adam_email'] : 'apcrouchley@gmail.com';

        $current_screen = get_current_screen();
        $current_page   = isset($current_screen->id) ? $current_screen->id : '';

        ?>
        <div class="adc-widget adc-widget-ask">
            <p class="adc-intro">
                Stuck, confused, or worried you’ll break something? Send a quick note to Adam and describe what you’re trying to do.
            </p>
            <p class="adc-small">
                Your message will be emailed to: <strong><?php echo esc_html($ask_email); ?></strong>
            </p>

            <form method="post" action="">
                <?php wp_nonce_field('adc_ask_adam', 'adc_ask_adam_nonce'); ?>
                <input type="hidden" name="adc_ask_adam_submit" value="1">
                <input type="hidden" name="adc_ask_adam_screen" value="<?php echo esc_attr($current_page); ?>">

                <p>
                    <label for="adc_ask_topic"><strong>What are you working on?</strong></label><br>
                    <select name="adc_ask_topic" id="adc_ask_topic" class="adc-select">
                        <option value="">Select a topic (optional)</option>
                        <option value="content">Writing or editing content</option>
                        <option value="design">Design / layout / Elementor</option>
                        <option value="forms">Contact forms / enquiries</option>
                        <option value="shop">Online shop / WooCommerce</option>
                        <option value="seo">SEO / Google visibility</option>
                        <option value="other">Something else</option>
                    </select>
                </p>

                <p>
                    <label for="adc_ask_message"><strong>Describe the problem in plain language</strong></label><br>
                    <textarea name="adc_ask_message" id="adc_ask_message" rows="4" class="adc-textarea"
                              placeholder="Example: I’m trying to change the main image on the homepage but the old one keeps showing."></textarea>
                </p>

                <p class="adc-small">
                    Adam will receive your name, email, and a link to this site so he knows where the question came from.
                </p>

                <p>
                    <button type="submit" class="button button-primary">Send to Adam</button>
                </p>
            </form>
        </div>
        <?php
    }

    /* ========================================================
     * Ask Adam form handling
     * ===================================================== */

    public function handle_ask_adam_form() {
        if (!is_admin() || !is_user_logged_in()) {
            return;
        }

        if (empty($_POST['adc_ask_adam_submit'])) {
            return;
        }

        if (!isset($_POST['adc_ask_adam_nonce']) || !wp_verify_nonce($_POST['adc_ask_adam_nonce'], 'adc_ask_adam')) {
            $this->set_ask_adam_notice('error', 'Security check failed. Please try again.');
            wp_safe_redirect(admin_url('index.php'));
            exit;
        }

        $message = isset($_POST['adc_ask_message']) ? trim(wp_kses_post(wp_unslash($_POST['adc_ask_message']))) : '';
        $topic   = isset($_POST['adc_ask_topic']) ? sanitize_text_field(wp_unslash($_POST['adc_ask_topic'])) : '';
        $screen  = isset($_POST['adc_ask_adam_screen']) ? sanitize_text_field(wp_unslash($_POST['adc_ask_adam_screen'])) : '';

        if ($message === '') {
            $this->set_ask_adam_notice('error', 'Please tell Adam a bit about what you need help with.');
            wp_safe_redirect(admin_url('index.php'));
            exit;
        }

        $current_user = wp_get_current_user();
        $opts         = $this->get_options();
        $to_email     = !empty($opts['ask_adam_email']) ? $opts['ask_adam_email'] : 'apcrouchley@gmail.com';

        $site_name = get_bloginfo('name');
        $site_url  = home_url('/');

        $topic_label = '';
        switch ($topic) {
            case 'content': $topic_label = 'Content'; break;
            case 'design':  $topic_label = 'Design / Layout'; break;
            case 'forms':   $topic_label = 'Forms / Enquiries'; break;
            case 'shop':    $topic_label = 'Shop / WooCommerce'; break;
            case 'seo':     $topic_label = 'SEO'; break;
            case 'other':   $topic_label = 'Other'; break;
        }

        $subject = sprintf('[%s] Ask Adam support request from %s', $site_name, $current_user->display_name);
        if ($topic_label) {
            $subject = '[' . $topic_label . '] ' . $subject;
        }

        $body_lines = array(
            'Site: ' . $site_name . ' (' . $site_url . ')',
            'User: ' . $current_user->display_name . ' (' . $current_user->user_email . ')',
        );

        if ($topic_label) {
            $body_lines[] = 'Topic: ' . $topic_label;
        }
        if ($screen) {
            $body_lines[] = 'Screen: ' . $screen;
        }

        $body_lines[] = '';
        $body_lines[] = 'Message:';
        $body_lines[] = $message;

        $body = implode("\n", $body_lines);

        $headers = array();
        if (!empty($current_user->user_email)) {
            $headers[] = 'Reply-To: ' . $current_user->display_name . ' <' . $current_user->user_email . '>';
        }

        $sent = wp_mail($to_email, $subject, $body, $headers);

        if ($sent) {
            $this->set_ask_adam_notice('success', 'Thanks – your message has been sent to Adam. He’ll get back to you as soon as he can.');
        } else {
            $this->set_ask_adam_notice('error', 'Something went wrong sending your message. Please try again or email your web support directly.');
        }

        wp_safe_redirect(admin_url('index.php'));
        exit;
    }

    private function set_ask_adam_notice($type, $message) {
        $user_id = get_current_user_id();
        if (!$user_id) {
            return;
        }
        set_transient('adc_ask_adam_notice_' . $user_id, array(
            'type'    => $type,
            'message' => $message,
        ), 60);
    }

    public function render_ask_adam_notice() {
        if (!is_admin()) {
            return;
        }

        global $pagenow;
        if ($pagenow !== 'index.php') {
            return;
        }

        $user_id = get_current_user_id();
        if (!$user_id) {
            return;
        }

        $notice = get_transient('adc_ask_adam_notice_' . $user_id);
        if (!$notice || empty($notice['message'])) {
            return;
        }

        delete_transient('adc_ask_adam_notice_' . $user_id);

        $class = 'notice-info';
        if ($notice['type'] === 'success') {
            $class = 'notice-success';
        } elseif ($notice['type'] === 'error') {
            $class = 'notice-error';
        }
        ?>
        <div class="notice <?php echo esc_attr($class); ?> is-dismissible">
            <p><?php echo esc_html($notice['message']); ?></p>
        </div>
        <?php
    }

    /* ========================================================
     * Menu tweaks
     * ===================================================== */

    public function filter_admin_menu() {
        $opts = $this->get_options();

        // Only apply hiding to users that CANNOT manage options
        if (current_user_can('manage_options')) {
            return;
        }

        global $menu;

        // Map of slug => option key
        $hide_map = array(
            'index.php'          => 'hide_dashboard',
            'edit-comments.php'  => 'hide_comments',
            'tools.php'          => 'hide_tools',
            'options-general.php'=> 'hide_settings',
            'plugins.php'        => 'hide_plugins',
            'themes.php'         => 'hide_appearance',
            'users.php'          => 'hide_users',
        );

        foreach ($menu as $index => $item) {
            if (!isset($item[2])) {
                continue;
            }
            $slug = $item[2];
            if (isset($hide_map[$slug])) {
                $opt_key = $hide_map[$slug];
                if (!empty($opts[$opt_key])) {
                    unset($menu[$index]);
                }
            }
        }
    }

    public function rename_admin_menus() {
        $opts = $this->get_options();

        global $menu;

        foreach ($menu as $index => $item) {
            if (!isset($item[0]) || !isset($item[2])) {
                continue;
            }

            $slug = $item[2];

            if (!empty($opts['rename_posts_to_blog']) && $slug === 'edit.php') {
                $menu[$index][0] = 'Blog';
            }

            if (!empty($opts['rename_media_to_library']) && $slug === 'upload.php') {
                $menu[$index][0] = 'Photos & Files';
            }

            if (!empty($opts['rename_pages_to_pages']) && $slug === 'edit.php?post_type=page') {
                $menu[$index][0] = 'Pages';
            }
        }
    }

    /* ========================================================
     * Settings page
     * ===================================================== */

    public function register_settings_page() {
        add_options_page(
            "Adam's Dashboard Cleaner",
            "Adam's Dashboard Cleaner",
            'manage_options',
            'adc-dashboard-cleaner',
            array($this, 'render_settings_page')
        );
    }

    public function register_settings() {
        register_setting(
            'adc_dashboard_cleaner_group',
            self::OPTION_KEY,
            array($this, 'sanitize_options')
        );

        // Sections
        add_settings_section(
            'adc_section_general',
            'General',
            function () {
                echo '<p>Set up a clean, friendly dashboard for your team. These options only affect the WordPress admin area.</p>';
            },
            'adc-dashboard-cleaner'
        );

        add_settings_section(
            'adc_section_menu',
            'Admin Menu (for non-admins)',
            function () {
                echo '<p>Hide items that regular Editors don’t need. These options do <strong>not</strong> affect Administrators.</p>';
            },
            'adc-dashboard-cleaner'
        );

        add_settings_section(
            'adc_section_labels',
            'Friendly Labels',
            function () {
                echo '<p>Rename some menu items to plain language. This affects all users.</p>';
            },
            'adc-dashboard-cleaner'
        );

        add_settings_section(
            'adc_section_support',
            'Ask Adam Settings',
            function () {
                echo '<p>Control where the “Ask Adam” support form sends messages.</p>';
            },
            'adc-dashboard-cleaner'
        );

        // General fields
        $this->add_checkbox_field('hide_default_dashboard_widgets', 'Hide default WordPress dashboard widgets', 'adc_section_general');
        $this->add_checkbox_field('add_start_here_widget', "Show “Adam’s Start Here” widget on Dashboard", 'adc_section_general');

        // Menu fields
        $this->add_checkbox_field('hide_dashboard', 'Hide Dashboard menu', 'adc_section_menu');
        $this->add_checkbox_field('hide_comments', 'Hide Comments', 'adc_section_menu');
        $this->add_checkbox_field('hide_tools', 'Hide Tools', 'adc_section_menu');
        $this->add_checkbox_field('hide_settings', 'Hide Settings', 'adc_section_menu');
        $this->add_checkbox_field('hide_plugins', 'Hide Plugins', 'adc_section_menu');
        $this->add_checkbox_field('hide_appearance', 'Hide Appearance', 'adc_section_menu');
        $this->add_checkbox_field('hide_users', 'Hide Users', 'adc_section_menu');

        // Labels fields
        $this->add_checkbox_field('rename_posts_to_blog', 'Rename “Posts” to “Blog”', 'adc_section_labels');
        $this->add_checkbox_field('rename_media_to_library', 'Rename “Media” to “Photos &amp; Files”', 'adc_section_labels');
        $this->add_checkbox_field('rename_pages_to_pages', 'Ensure “Pages” label is just “Pages”', 'adc_section_labels');

        // Support email field
        add_settings_field(
            'ask_adam_email',
            'Ask Adam email address',
            array($this, 'field_ask_adam_email'),
            'adc-dashboard-cleaner',
            'adc_section_support'
        );
    }

    private function add_checkbox_field($key, $label, $section) {
        add_settings_field(
            $key,
            $label,
            array($this, 'render_checkbox_field'),
            'adc-dashboard-cleaner',
            $section,
            array('key' => $key)
        );
    }

    public function sanitize_options($input) {
        $output = $this->get_options();

        $bool_keys = array(
            'hide_default_dashboard_widgets',
            'add_start_here_widget',
            'hide_comments',
            'hide_tools',
            'hide_settings',
            'hide_plugins',
            'hide_appearance',
            'hide_users',
            'hide_dashboard',
            'rename_posts_to_blog',
            'rename_media_to_library',
            'rename_pages_to_pages',
        );

        foreach ($bool_keys as $key) {
            $output[$key] = !empty($input[$key]) ? 1 : 0;
        }

        if (isset($input['ask_adam_email'])) {
            $email = sanitize_email($input['ask_adam_email']);
            if (!empty($email)) {
                $output['ask_adam_email'] = $email;
            }
        }

        return $output;
    }

    public function render_checkbox_field($args) {
        $key  = $args['key'];
        $opts = $this->get_options();
        ?>
        <label>
            <input type="checkbox"
                   name="<?php echo esc_attr(self::OPTION_KEY . '[' . $key . ']'); ?>"
                   value="1" <?php checked(!empty($opts[$key]), 1); ?>>
        </label>
        <?php
    }

    public function field_ask_adam_email() {
        $opts   = $this->get_options();
        $email  = isset($opts['ask_adam_email']) ? $opts['ask_adam_email'] : 'apcrouchley@gmail.com';
        ?>
        <input type="email"
               name="<?php echo esc_attr(self::OPTION_KEY); ?>[ask_adam_email]"
               value="<?php echo esc_attr($email); ?>"
               class="regular-text">
        <p class="description">
            All “Ask Adam” messages from the Dashboard will be emailed here.
        </p>
        <?php
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap adc-settings-wrap">
            <h1>Adam's Dashboard Cleaner</h1>
            <p class="description">
                Make the WordPress admin feel calm and obvious for your team.
                These settings only affect the backend; your public site is untouched.
            </p>

            <form method="post" action="options.php">
                <?php
                settings_fields('adc_dashboard_cleaner_group');
                do_settings_sections('adc-dashboard-cleaner');
                submit_button();
                ?>
            </form>

            <p class="description">
                Tip: log in as an Editor (or use a private/private window) to see what a normal user will actually see.
            </p>
        </div>
        <?php
    }

    /* ========================================================
     * CSS
     * ===================================================== */

    public function output_admin_css() {
        global $pagenow;

        // Only load CSS on Dashboard and plugin settings page
        $screen = get_current_screen();
        if ($pagenow !== 'index.php' && ($screen && $screen->id !== 'settings_page_adc-dashboard-cleaner')) {
            return;
        }
        ?>
        <style>
            .adc-widget,
            .adc-settings-wrap {
                font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            }
            .adc-widget-header h3 {
                margin-top: 0;
                font-size: 18px;
            }
            .adc-widget-header p {
                margin: 4px 0 12px;
                color: #6b7280;
                font-size: 13px;
            }
            .adc-widget-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
                gap: 12px;
            }
            .adc-card {
                background: #f9fafb;
                border-radius: 8px;
                border: 1px solid #e5e7eb;
                padding: 12px 12px 10px;
            }
            .adc-card h4 {
                margin: 0 0 6px;
                font-size: 14px;
            }
            .adc-card p {
                margin: 0 0 8px;
                font-size: 12px;
                color: #4b5563;
            }
            .adc-card-help {
                background: #eff6ff;
                border-color: #bfdbfe;
            }
            .adc-links-list {
                margin: 0 0 6px 16px;
                padding: 0;
                font-size: 12px;
            }
            .adc-links-list li {
                margin-bottom: 3px;
            }
            .adc-small {
                font-size: 11px;
                color: #6b7280;
                margin: 4px 0 0;
            }

            .adc-widget-ask .adc-intro {
                margin-top: 0;
                font-size: 13px;
                color: #4b5563;
            }
            .adc-widget-ask .adc-small {
                margin-top: 4px;
                margin-bottom: 10px;
            }
            .adc-widget-ask label {
                font-size: 13px;
            }
            .adc-select,
            .adc-textarea {
                width: 100%;
                max-width: 100%;
                border-radius: 6px;
                border: 1px solid #d1d5db;
                padding: 6px 8px;
                font-size: 13px;
            }
            .adc-textarea {
                min-height: 80px;
                resize: vertical;
            }
            .adc-settings-wrap .description {
                max-width: 720px;
            }
        </style>
        <?php
    }
}

ADC_Dashboard_Cleaner::get_instance();