<?php
/**
 * Plugin Name: Adam's Image Optimiser
 * Plugin URI:  https://plugins.dorimedia.co.nz/
 * Description: Automatic image resizing and optimisation for WordPress. Shrink existing images in your Media Library and free up disk space.
 * Version:     0.2.2
 * Author:      Adam / Dori Media
 * Author URI:  https://dorimedia.co.nz/
 * License:     GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: adams-image-optimiser
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'Adams_Image_Optimiser' ) ) :

class Adams_Image_Optimiser {

    const OPTION_KEY         = 'aio_settings';
    const OPTION_LAST_RUN    = 'aio_last_run';
    const OPTION_LAST_PURGE  = 'aio_last_purge';

    const NONCE_ACTION_RUN   = 'aio_run_optimisation';
    const NONCE_ACTION_PURGE = 'aio_purge_backups';
    const NONCE_AJAX         = 'aio_ajax_nonce';
    const NONCE_FIELD        = 'aio_nonce';

    const TRANSIENT_IDS_BASE   = 'aio_ids_';
    const TRANSIENT_STATS_BASE = 'aio_stats_';

    /**
     * Singleton instance
     * @var Adams_Image_Optimiser
     */
    protected static $instance = null;

    /**
     * Get singleton instance
     *
     * @return Adams_Image_Optimiser
     */
    public static function instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Defaults & settings
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_menu', array( $this, 'register_menu' ) );
        add_action( 'admin_notices', array( $this, 'admin_notices' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );

        // Optimise on upload
        add_filter( 'wp_handle_upload', array( $this, 'optimize_on_upload' ) );

        // Admin-post handler for purge
        add_action( 'admin_post_aio_purge_backups', array( $this, 'handle_purge_backups' ) );

        // AJAX handlers for batch optimisation
        add_action( 'wp_ajax_aio_prepare_optimisation', array( $this, 'ajax_prepare_optimisation' ) );
        add_action( 'wp_ajax_aio_process_batch', array( $this, 'ajax_process_batch' ) );
        add_action( 'wp_ajax_aio_finish_optimisation', array( $this, 'ajax_finish_optimisation' ) );
    }

    /**
     * Plugin activation hook
     */
    public static function activate() {
        $defaults = array(
            'max_width'        => 2560,
            'max_height'       => 2560,
            'quality'          => 82,
            'backup_originals' => 1,
            'force_recompress' => 0,
        );
        $saved = get_option( self::OPTION_KEY );
        if ( ! is_array( $saved ) ) {
            update_option( self::OPTION_KEY, $defaults );
        } else {
            update_option( self::OPTION_KEY, wp_parse_args( $saved, $defaults ) );
        }
    }

    /**
     * Get settings with defaults
     *
     * @return array
     */
    public function get_settings() {
        $defaults = array(
            'max_width'        => 2560,
            'max_height'       => 2560,
            'quality'          => 82,
            'backup_originals' => 1,
            'force_recompress' => 0,
        );
        $settings = get_option( self::OPTION_KEY, array() );
        if ( ! is_array( $settings ) ) {
            $settings = array();
        }
        return wp_parse_args( $settings, $defaults );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting(
            'aio_settings_group',
            self::OPTION_KEY,
            array( $this, 'sanitize_settings' )
        );

        add_settings_section(
            'aio_main_section',
            __( 'Image Resize Settings', 'adams-image-optimiser' ),
            function() {
                echo '<p>' . esc_html__( 'These limits apply to both new uploads and bulk optimisation. You can also force recompression to squeeze file sizes further.', 'adams-image-optimiser' ) . '</p>';
            },
            'aio_settings_page'
        );

        add_settings_field(
            'max_width',
            __( 'Maximum Width (px)', 'adams-image-optimiser' ),
            array( $this, 'field_max_width' ),
            'aio_settings_page',
            'aio_main_section'
        );

        add_settings_field(
            'max_height',
            __( 'Maximum Height (px)', 'adams-image-optimiser' ),
            array( $this, 'field_max_height' ),
            'aio_settings_page',
            'aio_main_section'
        );

        add_settings_field(
            'quality',
            __( 'JPEG Quality', 'adams-image-optimiser' ),
            array( $this, 'field_quality' ),
            'aio_settings_page',
            'aio_main_section'
        );

        add_settings_field(
            'backup_originals',
            __( 'Backup Originals', 'adams-image-optimiser' ),
            array( $this, 'field_backup_originals' ),
            'aio_settings_page',
            'aio_main_section'
        );

        add_settings_field(
            'force_recompress',
            __( 'Force Recompression', 'adams-image-optimiser' ),
            array( $this, 'field_force_recompress' ),
            'aio_settings_page',
            'aio_main_section'
        );
    }

    /**
     * Sanitize settings
     */
    public function sanitize_settings( $input ) {
        $output = $this->get_settings();

        if ( isset( $input['max_width'] ) ) {
            $output['max_width'] = max( 0, intval( $input['max_width'] ) );
        }

        if ( isset( $input['max_height'] ) ) {
            $output['max_height'] = max( 0, intval( $input['max_height'] ) );
        }

        if ( isset( $input['quality'] ) ) {
            $q = intval( $input['quality'] );
            if ( $q < 10 ) {
                $q = 10;
            } elseif ( $q > 100 ) {
                $q = 100;
            }
            $output['quality'] = $q;
        }

        $output['backup_originals'] = ! empty( $input['backup_originals'] ) ? 1 : 0;
        $output['force_recompress'] = ! empty( $input['force_recompress'] ) ? 1 : 0;

        add_settings_error(
            'aio_messages',
            'aio_settings_saved',
            __( 'Adam\'s Image Optimiser settings saved.', 'adams-image-optimiser' ),
            'updated'
        );

        return $output;
    }

    /**
     * Settings fields renderers
     */
    public function field_max_width() {
        $settings = $this->get_settings();
        printf(
            '<input type="number" name="%1$s[max_width]" value="%2$d" min="0" class="small-text" /> <p class="description">%3$s</p>',
            esc_attr( self::OPTION_KEY ),
            intval( $settings['max_width'] ),
            esc_html__( 'Images wider than this will be resized down on upload and during bulk optimisation. Set to 0 to disable width limit.', 'adams-image-optimiser' )
        );
    }

    public function field_max_height() {
        $settings = $this->get_settings();
        printf(
            '<input type="number" name="%1$s[max_height]" value="%2$d" min="0" class="small-text" /> <p class="description">%3$s</p>',
            esc_attr( self::OPTION_KEY ),
            intval( $settings['max_height'] ),
            esc_html__( 'Images taller than this will be resized down. Set to 0 to disable height limit.', 'adams-image-optimiser' )
        );
    }

    public function field_quality() {
        $settings = $this->get_settings();
        printf(
            '<input type="number" name="%1$s[quality]" value="%2$d" min="10" max="100" class="small-text" /> <p class="description">%3$s</p>',
            esc_attr( self::OPTION_KEY ),
            intval( $settings['quality'] ),
            esc_html__( 'JPEG quality for resized images. 80–85 is a good balance between size and clarity; lower = smaller files.', 'adams-image-optimiser' )
        );
    }

    public function field_backup_originals() {
        $settings = $this->get_settings();
        printf(
            '<label><input type="checkbox" name="%1$s[backup_originals]" value="1" %2$s /> %3$s</label><p class="description">%4$s</p>',
            esc_attr( self::OPTION_KEY ),
            checked( $settings['backup_originals'], 1, false ),
            esc_html__( 'Store a backup copy of originals before resizing.', 'adams-image-optimiser' ),
            esc_html__( 'Backups are stored in wp-content/uploads/adams-image-optimiser-originals/. Disable this for future runs if you are confident, or use the Purge Backups tool to remove old copies.', 'adams-image-optimiser' )
        );
    }

    public function field_force_recompress() {
        $settings = $this->get_settings();
        printf(
            '<label><input type="checkbox" name="%1$s[force_recompress]" value="1" %2$s /> %3$s</label><p class="description">%4$s</p>',
            esc_attr( self::OPTION_KEY ),
            checked( $settings['force_recompress'], 1, false ),
            esc_html__( 'Always recompress images even if they do not exceed the max width/height.', 'adams-image-optimiser' ),
            esc_html__( 'When enabled, every JPEG/PNG will be re-saved using your quality setting during bulk runs and CLI. This squeezes more space savings but will take longer.', 'adams-image-optimiser' )
        );
    }

    /**
     * Register admin menu
     */
    public function register_menu() {
        add_submenu_page(
            'options-general.php',
            __( 'Adam\'s Image Optimiser', 'adams-image-optimiser' ),
            __( 'Adam\'s Image Optimiser', 'adams-image-optimiser' ),
            'manage_options',
            'adams-image-optimiser',
            array( $this, 'render_settings_page' )
        );
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_assets( $hook ) {
        if ( 'settings_page_adams-image-optimiser' !== $hook ) {
            return;
        }

        wp_enqueue_script(
            'aio-admin',
            plugins_url( 'assets/aio-admin.js', __FILE__ ),
            array( 'jquery' ),
            '0.2.2',
            true
        );

        wp_localize_script(
            'aio-admin',
            'AIOSettings',
            array(
                'ajax_url' => admin_url( 'admin-ajax.php' ),
                'nonce'    => wp_create_nonce( self::NONCE_AJAX ),
            )
        );
    }

    /**
     * Render settings & tools page
     */
    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $settings = $this->get_settings();
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Adam\'s Image Optimiser', 'adams-image-optimiser' ); ?></h1>
            <p><?php esc_html_e( 'Automatic image resizing and optimisation. Shrink existing images in your Media Library and free up disk space.', 'adams-image-optimiser' ); ?></p>

            <h2 class="nav-tab-wrapper">
                <a href="<?php echo esc_url( admin_url( 'options-general.php?page=adams-image-optimiser' ) ); ?>" class="nav-tab nav-tab-active"><?php esc_html_e( 'Settings &amp; Tools', 'adams-image-optimiser' ); ?></a>
            </h2>

            <div class="aio-columns" style="display:flex; gap:40px; align-items:flex-start;">
                <div class="aio-main" style="flex:2;">
                    <form method="post" action="options.php">
                        <?php
                        settings_fields( 'aio_settings_group' );
                        do_settings_sections( 'aio_settings_page' );
                        submit_button( __( 'Save Settings', 'adams-image-optimiser' ) );
                        ?>
                    </form>

                    <hr />

                    <h2><?php esc_html_e( 'Resize Existing Images', 'adams-image-optimiser' ); ?></h2>
                    <p><?php esc_html_e( 'Run a bulk pass over your Media Library. Images larger than your max width/height will be resized and overwritten, and other images can be recompressed if you choose.', 'adams-image-optimiser' ); ?></p>
                    <p><strong><?php esc_html_e( 'Tip:', 'adams-image-optimiser' ); ?></strong> <?php esc_html_e( 'Use a dry run first to see how many images would be updated before touching any files.', 'adams-image-optimiser' ); ?></p>

                    <p>
                        <label>
                            <input type="checkbox" id="aio-dry-run" value="1" />
                            <?php esc_html_e( 'Dry run only (analyse, don’t modify any files).', 'adams-image-optimiser' ); ?>
                        </label>
                        <br />
                        <span class="description"><?php esc_html_e( 'Dry run will still tell you how many images would be resized/recompressed, but no backups are created and nothing is written to disk.', 'adams-image-optimiser' ); ?></span>
                    </p>

                    <div id="aio-inline-message" style="margin:10px 0;"></div>

                    <button id="aio-run-button" class="button button-primary">
                        <?php esc_html_e( 'Resize &amp; Optimise Images Now', 'adams-image-optimiser' ); ?>
                    </button>
                    <p class="description"><?php esc_html_e( 'Leave this tab open while the progress bar runs. You can switch to other tabs in your browser while it works.', 'adams-image-optimiser' ); ?></p>

                    <div id="aio-progress-wrapper" style="margin-top:15px; display:none; max-width:480px;">
                        <progress id="aio-progress-bar" max="100" value="0" style="width:100%; height:20px;"></progress>
                        <p id="aio-progress-text" style="margin-top:8px;"><?php esc_html_e( 'Preparing…', 'adams-image-optimiser' ); ?></p>
                    </div>
                </div>

                <div class="aio-sidebar" style="flex:1; background:#fff; padding:20px; border:1px solid #ccd0d4; border-radius:4px;">
                    <h2><?php esc_html_e( 'Status', 'adams-image-optimiser' ); ?></h2>
                    <?php $this->render_status_box(); ?>

                    <hr />

                    <h3><?php esc_html_e( 'Purge Backup Originals', 'adams-image-optimiser' ); ?></h3>
                    <p><?php esc_html_e( 'Once you are confident that the resized images look good, you can delete the backup copies of the originals to free up server space.', 'adams-image-optimiser' ); ?></p>
                    <?php
                    $backup_size = $this->get_backup_dir_size();
                    if ( $backup_size > 0 ) {
                        echo '<p><strong>' . esc_html__( 'Current backup size:', 'adams-image-optimiser' ) . '</strong> ' . esc_html( $this->format_bytes( $backup_size ) ) . '</p>';
                    } else {
                        echo '<p><em>' . esc_html__( 'No backup originals found. Either backups are disabled or you have already purged them.', 'adams-image-optimiser' ) . '</em></p>';
                    }
                    ?>
                    <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" onsubmit="return confirm('<?php echo esc_js( __( 'Are you sure you want to permanently delete all backup originals created by this plugin?', 'adams-image-optimiser' ) ); ?>');">
                        <?php wp_nonce_field( self::NONCE_ACTION_PURGE, self::NONCE_FIELD ); ?>
                        <input type="hidden" name="action" value="aio_purge_backups" />
                        <?php submit_button( __( 'Purge Backups', 'adams-image-optimiser' ), 'secondary', 'aio_purge', false ); ?>
                    </form>

                    <p style="margin-top:1em;font-size:12px;color:#555;">
                        <?php esc_html_e( 'Note: This plugin focuses on safe resizing and compression. It does not yet rewrite image URLs on the front-end to WebP – that will come in a future update.', 'adams-image-optimiser' ); ?>
                    </p>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render sidebar status box
     */
    protected function render_status_box() {
        $last = get_option( self::OPTION_LAST_RUN, array() );
        if ( ! empty( $last['time'] ) ) {
            echo '<p><strong>' . esc_html__( 'Last real run:', 'adams-image-optimiser' ) . '</strong><br />' . esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), intval( $last['time'] ) ) ) . '</p>';
            echo '<p><strong>' . esc_html__( 'Images checked:', 'adams-image-optimiser' ) . '</strong> ' . intval( $last['checked'] ) . '</p>';
            echo '<p><strong>' . esc_html__( 'Images updated:', 'adams-image-optimiser' ) . '</strong> ' . intval( $last['resized'] ) . '</p>';
            echo '<p><strong>' . esc_html__( 'Space saved:', 'adams-image-optimiser' ) . '</strong> ' . esc_html( $this->format_bytes( isset( $last['bytes_saved'] ) ? $last['bytes_saved'] : 0 ) ) . '</p>';
            echo '<p class="description">' . esc_html__( 'Dry runs do not update these stats – they only analyse.', 'adams-image-optimiser' ) . '</p>';
        } else {
            echo '<p>' . esc_html__( 'No optimisation runs yet (or last run was a dry run).', 'adams-image-optimiser' ) . '</p>';
        }

        $settings = $this->get_settings();
        echo '<hr />';
        echo '<p><strong>' . esc_html__( 'Current limits & behaviour:', 'adams-image-optimiser' ) . '</strong><br />';
        echo esc_html__( 'Max width:', 'adams-image-optimiser' ) . ' ' . ( $settings['max_width'] ? intval( $settings['max_width'] ) . 'px' : esc_html__( 'no limit', 'adams-image-optimiser' ) ) . '<br />';
        echo esc_html__( 'Max height:', 'adams-image-optimiser' ) . ' ' . ( $settings['max_height'] ? intval( $settings['max_height'] ) . 'px' : esc_html__( 'no limit', 'adams-image-optimiser' ) ) . '<br />';
        echo esc_html__( 'JPEG quality:', 'adams-image-optimiser' ) . ' ' . intval( $settings['quality'] ) . '<br />';
        echo esc_html__( 'Force recompression:', 'adams-image-optimiser' ) . ' ' . ( $settings['force_recompress'] ? esc_html__( 'enabled – all images will be re-saved.', 'adams-image-optimiser' ) : esc_html__( 'disabled – only oversize images are changed.', 'adams-image-optimiser' ) );
        echo '</p>';

        $last_purge = get_option( self::OPTION_LAST_PURGE, array() );
        if ( ! empty( $last_purge['time'] ) ) {
            echo '<hr />';
            echo '<p><strong>' . esc_html__( 'Last purge:', 'adams-image-optimiser' ) . '</strong><br />' . esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), intval( $last_purge['time'] ) ) ) . '</p>';
            echo '<p><strong>' . esc_html__( 'Space freed:', 'adams-image-optimiser' ) . '</strong> ' . esc_html( $this->format_bytes( isset( $last_purge['bytes_deleted'] ) ? $last_purge['bytes_deleted'] : 0 ) ) . '</p>';
        }
    }

    /**
     * Handle admin notices
     */
    public function admin_notices() {
        settings_errors( 'aio_messages' );

        if ( isset( $_GET['aio_purged'] ) && '1' === $_GET['aio_purged'] ) {
            $last_purge = get_option( self::OPTION_LAST_PURGE, array() );
            ?>
            <div class="notice notice-success is-dismissible">
                <p>
                    <?php
                    printf(
                        esc_html__( 'Adam\'s Image Optimiser purge completed. Freed approximately %s of disk space.', 'adams-image-optimiser' ),
                        esc_html( $this->format_bytes( isset( $last_purge['bytes_deleted'] ) ? $last_purge['bytes_deleted'] : 0 ) )
                    );
                    ?>
                </p>
            </div>
            <?php
        }
    }

    /**
     * Optimise image on upload
     *
     * @param array $upload
     * @return array
     */
    public function optimize_on_upload( $upload ) {
        if ( empty( $upload['file'] ) ) {
            return $upload;
        }

        $file = $upload['file'];

        // Ensure it's an image
        $mime = isset( $upload['type'] ) ? $upload['type'] : '';
        if ( strpos( $mime, 'image/' ) !== 0 ) {
            return $upload;
        }

        $settings = $this->get_settings();

        $this->maybe_resize_image_file( $file, $settings, false, false );

        return $upload;
    }

    /**
     * AJAX: prepare optimisation run (collect attachment IDs)
     */
    public function ajax_prepare_optimisation() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'adams-image-optimiser' ) ) );
        }

        check_ajax_referer( self::NONCE_AJAX, 'nonce' );

        $args = array(
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'posts_per_page' => -1,
            'fields'         => 'ids',
        );

        $query = new WP_Query( $args );
        $ids   = $query->posts;

        $ids_key   = $this->get_ids_transient_key();
        $stats_key = $this->get_stats_transient_key();

        set_transient( $ids_key, $ids, 12 * HOUR_IN_SECONDS );
        set_transient(
            $stats_key,
            array(
                'checked'     => 0,
                'resized'     => 0,
                'bytes_saved' => 0,
            ),
            12 * HOUR_IN_SECONDS
        );

        wp_send_json_success(
            array(
                'total' => count( $ids ),
            )
        );
    }

    /**
     * AJAX: process a batch of images
     */
    public function ajax_process_batch() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'adams-image-optimiser' ) ) );
        }

        check_ajax_referer( self::NONCE_AJAX, 'nonce' );

        $start   = isset( $_POST['start'] ) ? intval( $_POST['start'] ) : 0;
        $batch   = isset( $_POST['batch'] ) ? intval( $_POST['batch'] ) : 25;
        $dry_run = ! empty( $_POST['dry_run'] );

        if ( $batch <= 0 ) {
            $batch = 25;
        }
        if ( $batch > 200 ) {
            $batch = 200;
        }

        $ids_key   = $this->get_ids_transient_key();
        $stats_key = $this->get_stats_transient_key();

        $ids = get_transient( $ids_key );
        if ( ! is_array( $ids ) ) {
            wp_send_json_error( array( 'message' => __( 'No image list found. Please refresh the page and try again.', 'adams-image-optimiser' ) ) );
        }

        $total = count( $ids );

        $settings = $this->get_settings();

        $checked     = 0;
        $resized     = 0;
        $bytes_saved = 0;

        for ( $i = $start; $i < min( $start + $batch, $total ); $i++ ) {
            $attachment_id = $ids[ $i ];
            $file          = get_attached_file( $attachment_id );

            if ( ! $file || ! file_exists( $file ) ) {
                $checked++;
                continue;
            }

            $before_size = filesize( $file );
            $changed     = $this->maybe_resize_image_file( $file, $settings, ! $dry_run, $dry_run );

            $checked++;

            if ( $changed && ! $dry_run ) {
                $after_size = file_exists( $file ) ? filesize( $file ) : $before_size;
                if ( $after_size < $before_size ) {
                    $bytes_saved += ( $before_size - $after_size );
                }

                // Regenerate metadata so sizes are in sync
                if ( function_exists( 'wp_generate_attachment_metadata' ) ) {
                    $meta = wp_generate_attachment_metadata( $attachment_id, $file );
                    if ( ! is_wp_error( $meta ) && ! empty( $meta ) ) {
                        wp_update_attachment_metadata( $attachment_id, $meta );
                    }
                }
            }

            if ( $changed ) {
                $resized++;
            }
        }

        // Update stats transient
        $stats = get_transient( $stats_key );
        if ( ! is_array( $stats ) ) {
            $stats = array(
                'checked'     => 0,
                'resized'     => 0,
                'bytes_saved' => 0,
            );
        }

        $stats['checked']     += $checked;
        $stats['resized']     += $resized;
        $stats['bytes_saved'] += $bytes_saved;

        set_transient( $stats_key, $stats, 12 * HOUR_IN_SECONDS );

        $next_start = $start + $batch;

        wp_send_json_success(
            array(
                'next_start'  => $next_start,
                'total'       => $total,
                'checked'     => $stats['checked'],
                'resized'     => $stats['resized'],
                'bytes_saved' => $stats['bytes_saved'],
                'finished'    => ( $next_start >= $total ),
            )
        );
    }

    /**
     * AJAX: finish optimisation run (persist stats & cleanup)
     */
    public function ajax_finish_optimisation() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'adams-image-optimiser' ) ) );
        }

        check_ajax_referer( self::NONCE_AJAX, 'nonce' );

        $dry_run   = ! empty( $_POST['dry_run'] );
        $stats_key = $this->get_stats_transient_key();
        $ids_key   = $this->get_ids_transient_key();

        $stats = get_transient( $stats_key );
        if ( ! is_array( $stats ) ) {
            $stats = array(
                'checked'     => 0,
                'resized'     => 0,
                'bytes_saved' => 0,
            );
        }

        delete_transient( $stats_key );
        delete_transient( $ids_key );

        if ( ! $dry_run ) {
            $last_run = array(
                'time'        => time(),
                'checked'     => intval( $stats['checked'] ),
                'resized'     => intval( $stats['resized'] ),
                'bytes_saved' => intval( $stats['bytes_saved'] ),
            );

            update_option( self::OPTION_LAST_RUN, $last_run );
        }

        wp_send_json_success(
            array(
                'checked'           => intval( $stats['checked'] ),
                'resized'           => intval( $stats['resized'] ),
                'bytes_saved'       => intval( $stats['bytes_saved'] ),
                'bytes_saved_human' => $this->format_bytes( $stats['bytes_saved'] ),
                'dry_run'           => $dry_run,
            )
        );
    }

    /**
     * Handle purge backups action
     */
    public function handle_purge_backups() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to do this.', 'adams-image-optimiser' ) );
        }

        check_admin_referer( self::NONCE_ACTION_PURGE, self::NONCE_FIELD );

        $bytes_deleted = $this->purge_backups_dir();

        update_option(
            self::OPTION_LAST_PURGE,
            array(
                'time'          => time(),
                'bytes_deleted' => $bytes_deleted,
            )
        );

        wp_safe_redirect(
            add_query_arg(
                array(
                    'page'       => 'adams-image-optimiser',
                    'aio_purged' => '1',
                ),
                admin_url( 'options-general.php' )
            )
        );
        exit;
    }

    /**
     * Resize & compress a single image file if it exceeds limits or recompress is forced
     *
     * @param string $file_path
     * @param array  $settings
     * @param bool   $allow_backup
     * @param bool   $dry_run
     *
     * @return bool True if file was (or would be) changed
     */
    protected function maybe_resize_image_file( $file_path, $settings, $allow_backup = true, $dry_run = false ) {
        if ( ! file_exists( $file_path ) ) {
            return false;
        }

        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return false;
        }

        $width  = isset( $image_info[0] ) ? intval( $image_info[0] ) : 0;
        $height = isset( $image_info[1] ) ? intval( $image_info[1] ) : 0;
        $mime   = isset( $image_info['mime'] ) ? $image_info['mime'] : '';

        $max_width        = isset( $settings['max_width'] ) ? intval( $settings['max_width'] ) : 0;
        $max_height       = isset( $settings['max_height'] ) ? intval( $settings['max_height'] ) : 0;
        $quality          = isset( $settings['quality'] ) ? intval( $settings['quality'] ) : 82;
        $force_recompress = ! empty( $settings['force_recompress'] );

        // Only handle common raster images
        if ( ! in_array( $mime, array( 'image/jpeg', 'image/png', 'image/gif' ), true ) ) {
            return false;
        }

        // Determine if resize is needed
        $needs_resize = false;

        if ( $max_width > 0 && $width > $max_width ) {
            $needs_resize = true;
        }

        if ( $max_height > 0 && $height > $max_height ) {
            $needs_resize = true;
        }

        $needs_change = $needs_resize || $force_recompress;

        if ( ! $needs_change ) {
            return false;
        }

        // For dry run we only report that we *would* change this image
        if ( $dry_run ) {
            return true;
        }

        // Backup original if requested
        if ( $allow_backup && ! empty( $settings['backup_originals'] ) ) {
            $this->backup_original( $file_path );
        }

        $editor = wp_get_image_editor( $file_path );
        if ( is_wp_error( $editor ) ) {
            return false;
        }

        $resize_width  = $max_width > 0 ? $max_width : $width;
        $resize_height = $max_height > 0 ? $max_height : $height;

        // Maintain aspect ratio (crop = false). If no resize is needed but recompress is forced,
        // resize to the existing dimensions to trigger a save with new quality.
        if ( ! $needs_resize && $force_recompress ) {
            $resize_width  = $width;
            $resize_height = $height;
        }

        $result = $editor->resize( $resize_width, $resize_height, false );
        if ( is_wp_error( $result ) ) {
            return false;
        }

        if ( method_exists( $editor, 'set_quality' ) ) {
            $editor->set_quality( $quality );
        }

        $saved = $editor->save( $file_path );
        if ( is_wp_error( $saved ) ) {
            return false;
        }

        return true;
    }

    /**
     * Backup original image file
     *
     * @param string $file_path
     */
    protected function backup_original( $file_path ) {
        $upload_dir = wp_upload_dir();
        if ( empty( $upload_dir['basedir'] ) ) {
            return;
        }

        $relative = str_replace( $upload_dir['basedir'], '', $file_path );
        $relative = ltrim( $relative, '/\\' );

        $backup_base = trailingslashit( $upload_dir['basedir'] ) . 'adams-image-optimiser-originals';

        $backup_path = trailingslashit( $backup_base ) . $relative;
        $backup_dir  = dirname( $backup_path );

        if ( ! file_exists( $backup_dir ) ) {
            wp_mkdir_p( $backup_dir );
        }

        if ( file_exists( $backup_path ) ) {
            // Backup already exists, do not overwrite
            return;
        }

        @copy( $file_path, $backup_path );
    }

    /**
     * Purge backup directory
     *
     * @return int Bytes deleted
     */
    protected function purge_backups_dir() {
        $backup_base = $this->get_backup_base_dir();
        if ( ! $backup_base || ! file_exists( $backup_base ) ) {
            return 0;
        }

        $bytes_deleted = 0;

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $backup_base, RecursiveDirectoryIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ( $iterator as $fileinfo ) {
            /** @var SplFileInfo $fileinfo */
            if ( $fileinfo->isFile() ) {
                $bytes_deleted += $fileinfo->getSize();
                @unlink( $fileinfo->getRealPath() );
            } elseif ( $fileinfo->isDir() ) {
                @rmdir( $fileinfo->getRealPath() );
            }
        }

        // Remove the base directory itself
        @rmdir( $backup_base );

        return $bytes_deleted;
    }

    /**
     * Get backup base directory path
     *
     * @return string
     */
    protected function get_backup_base_dir() {
        $upload_dir = wp_upload_dir();
        if ( empty( $upload_dir['basedir'] ) ) {
            return '';
        }
        return trailingslashit( $upload_dir['basedir'] ) . 'adams-image-optimiser-originals';
    }

    /**
     * Calculate backup directory size
     *
     * @return int
     */
    protected function get_backup_dir_size() {
        $backup_base = $this->get_backup_base_dir();
        if ( ! $backup_base || ! file_exists( $backup_base ) ) {
            return 0;
        }

        $size = 0;

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $backup_base, RecursiveDirectoryIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ( $iterator as $fileinfo ) {
            if ( $fileinfo->isFile() ) {
                $size += $fileinfo->getSize();
            }
        }

        return $size;
    }

    /**
     * Format bytes as human readable string
     *
     * @param int $bytes
     * @return string
     */
    public function format_bytes( $bytes ) {
        $bytes = (int) $bytes;
        if ( $bytes <= 0 ) {
            return '0 B';
        }

        $units = array( 'B', 'KB', 'MB', 'GB', 'TB' );
        $power = floor( log( $bytes, 1024 ) );
        $power = min( $power, count( $units ) - 1 );

        $value = $bytes / pow( 1024, $power );

        return sprintf( '%.2f %s', $value, $units[ $power ] );
    }

    /**
     * Get transient key for IDs for current user
     */
    protected function get_ids_transient_key() {
        return self::TRANSIENT_IDS_BASE . get_current_user_id();
    }

    /**
     * Get transient key for stats for current user
     */
    protected function get_stats_transient_key() {
        return self::TRANSIENT_STATS_BASE . get_current_user_id();
    }
}

// Initialize plugin
add_action( 'plugins_loaded', array( 'Adams_Image_Optimiser', 'instance' ) );

// Activation hook
register_activation_hook( __FILE__, array( 'Adams_Image_Optimiser', 'activate' ) );

/**
 * WP-CLI integration
 */
if ( defined( 'WP_CLI' ) && WP_CLI && class_exists( 'Adams_Image_Optimiser' ) ) {

    /**
     * Adam's Image Optimiser CLI commands.
     */
    class Adams_Image_Optimiser_CLI extends WP_CLI_Command {

        /**
         * Resize and/or recompress images in the Media Library.
         *
         * ## OPTIONS
         *
         * [--max-width=<pixels>]
         * : Maximum width. Overrides plugin setting for this run.
         *
         * [--max-height=<pixels>]
         * : Maximum height. Overrides plugin setting for this run.
         *
         * [--quality=<percent>]
         * : JPEG quality (10-100). Overrides plugin setting for this run.
         *
         * [--batch=<number>]
         * : Number of images to process per page. Default 100.
         *
         * [--dry-run]
         * : Analyse only. Do not modify any files.
         *
         * ## EXAMPLES
         *
         *     wp adams-image-optimiser resize --max-width=2560 --max-height=2560 --quality=82 --batch=100
         *
         * @subcommand resize
         */
        public function resize( $args, $assoc_args ) {
            $plugin   = Adams_Image_Optimiser::instance();
            $settings = $plugin->get_settings();

            if ( isset( $assoc_args['max-width'] ) ) {
                $settings['max_width'] = intval( $assoc_args['max-width'] );
            }
            if ( isset( $assoc_args['max-height'] ) ) {
                $settings['max_height'] = intval( $assoc_args['max-height'] );
            }
            if ( isset( $assoc_args['quality'] ) ) {
                $settings['quality'] = intval( $assoc_args['quality'] );
            }

            $dry_run = isset( $assoc_args['dry-run'] );
            $batch   = isset( $assoc_args['batch'] ) ? max( 1, intval( $assoc_args['batch'] ) ) : 100;

            $args = array(
                'post_type'      => 'attachment',
                'post_status'    => 'inherit',
                'posts_per_page' => $batch,
                'fields'         => 'ids',
                'paged'          => 1,
            );

            $query = new WP_Query( $args );
            $total = $query->found_posts;

            if ( 0 === $total ) {
                WP_CLI::success( 'No attachments found.' );
                return;
            }

            $label = $dry_run ? 'Analysing images (dry run)' : 'Optimising images';
            $progress = \WP_CLI\Utils\make_progress_bar( $label, $total );

            $checked    = 0;
            $resized    = 0;
            $bytes_saved = 0;
            $page       = 1;

            while ( true ) {
                $args['paged'] = $page;
                $query         = new WP_Query( $args );

                if ( ! $query->have_posts() ) {
                    break;
                }

                foreach ( $query->posts as $attachment_id ) {
                    $checked++;

                    $file = get_attached_file( $attachment_id );
                    if ( ! $file || ! file_exists( $file ) ) {
                        $progress->tick();
                        continue;
                    }

                    $before_size = filesize( $file );
                    $changed     = $plugin->maybe_resize_image_file( $file, $settings, ! $dry_run, $dry_run );

                    if ( $changed && ! $dry_run ) {
                        $after_size = file_exists( $file ) ? filesize( $file ) : $before_size;
                        if ( $after_size < $before_size ) {
                            $bytes_saved += ( $before_size - $after_size );
                        }

                        if ( function_exists( 'wp_generate_attachment_metadata' ) ) {
                            $meta = wp_generate_attachment_metadata( $attachment_id, $file );
                            if ( ! is_wp_error( $meta ) && ! empty( $meta ) ) {
                                wp_update_attachment_metadata( $attachment_id, $meta );
                            }
                        }
                    }

                    if ( $changed ) {
                        $resized++;
                    }

                    $progress->tick();
                }

                $page++;
            }

            $progress->finish();

            if ( ! $dry_run ) {
                update_option(
                    Adams_Image_Optimiser::OPTION_LAST_RUN,
                    array(
                        'time'        => time(),
                        'checked'     => $checked,
                        'resized'     => $resized,
                        'bytes_saved' => $bytes_saved,
                    )
                );

                WP_CLI::success(
                    sprintf(
                        'Finished. Checked %1$d images, updated %2$d, saved approximately %3$s of disk space.',
                        $checked,
                        $resized,
                        $plugin->format_bytes( $bytes_saved )
                    )
                );
            } else {
                WP_CLI::success(
                    sprintf(
                        'Dry run complete. Checked %1$d images. %2$d images would be updated (resized and/or recompressed).',
                        $checked,
                        $resized
                    )
                );
            }
        }

        /**
         * Purge backup originals folder.
         *
         * ## EXAMPLES
         *
         *     wp adams-image-optimiser purge-backups
         *
         * @subcommand purge-backups
         */
        public function purge_backups( $args, $assoc_args ) {
            $plugin = Adams_Image_Optimiser::instance();

            $bytes_deleted = $plugin->purge_backups_dir();

            update_option(
                Adams_Image_Optimiser::OPTION_LAST_PURGE,
                array(
                    'time'          => time(),
                    'bytes_deleted' => $bytes_deleted,
                )
            );

            WP_CLI::success(
                sprintf(
                    'Purged backup originals and freed approximately %s of disk space.',
                    $plugin->format_bytes( $bytes_deleted )
                )
            );
        }
    }

    WP_CLI::add_command( 'adams-image-optimiser', 'Adams_Image_Optimiser_CLI' );
}

endif;
