jQuery(function($) {
    var running   = false;
    var total     = 0;
    var checked   = 0;
    var resized   = 0;
    var bytesSaved = 0;
    var batchSize = 25;
    var dryRun    = false;

    var $runButton   = $('#aio-run-button');
    var $progressWrap= $('#aio-progress-wrapper');
    var $progressBar = $('#aio-progress-bar');
    var $progressText= $('#aio-progress-text');
    var $message     = $('#aio-inline-message');
    var $dryRunBox   = $('#aio-dry-run');

    function setMessage(type, text) {
        if (!$message.length) {
            return;
        }
        var cls = 'notice ';
        if (type === 'error') {
            cls += 'notice-error';
        } else if (type === 'success') {
            cls += 'notice-success';
        } else if (type === 'info') {
            cls += 'notice-info';
        } else {
            cls += 'notice-info';
        }
        $message
            .removeClass()
            .addClass(cls)
            .html(text ? '<p>' + text + '</p>' : '');
    }

    function humanBytes(bytes) {
        bytes = parseInt(bytes, 10) || 0;
        if (bytes <= 0) return '0 B';
        var units = ['B', 'KB', 'MB', 'GB', 'TB'];
        var power = Math.floor(Math.log(bytes) / Math.log(1024));
        power = Math.min(power, units.length - 1);
        var value = bytes / Math.pow(1024, power);
        return value.toFixed(2) + ' ' + units[power];
    }

    function prepareRun() {
        running = true;
        total = 0;
        checked = 0;
        resized = 0;
        bytesSaved = 0;
        dryRun = $dryRunBox.is(':checked');

        $progressWrap.show();
        $progressBar.val(0);
        $progressText.text('Preparing…');
        setMessage('info', dryRun
            ? 'Dry run in progress. We\'re just analysing which images would change – no files will be touched.'
            : 'Optimisation in progress. Images may be resized and/or recompressed – keep this tab open while it runs.'
        );

        $.post(AIOSettings.ajax_url, {
            action: 'aio_prepare_optimisation',
            nonce: AIOSettings.nonce
        }).done(function(resp) {
            if (!resp || !resp.success) {
                running = false;
                var msg = (resp && resp.data && resp.data.message) ? resp.data.message : 'Error preparing optimisation.';
                setMessage('error', msg);
                return;
            }

            total = parseInt(resp.data.total, 10) || 0;
            if (total === 0) {
                running = false;
                $progressText.text('No images found in the Media Library.');
                setMessage('info', 'No attachments found. Nothing to analyse.');
                return;
            }

            $progressText.text('Starting… 0 / ' + total);
            processBatch(0);
        }).fail(function() {
            running = false;
            setMessage('error', 'AJAX error while preparing optimisation.');
        });
    }

    function processBatch(start) {
        if (!running) return;

        $.post(AIOSettings.ajax_url, {
            action: 'aio_process_batch',
            nonce: AIOSettings.nonce,
            start: start,
            batch: batchSize,
            dry_run: dryRun ? 1 : 0
        }).done(function(resp) {
            if (!resp || !resp.success) {
                running = false;
                var msg = (resp && resp.data && resp.data.message) ? resp.data.message : 'Error during optimisation.';
                setMessage('error', msg);
                return;
            }

            total      = parseInt(resp.data.total, 10) || total;
            checked    = parseInt(resp.data.checked, 10) || checked;
            resized    = parseInt(resp.data.resized, 10) || resized;
            bytesSaved = parseInt(resp.data.bytes_saved, 10) || bytesSaved;

            var percent = total > 0 ? Math.round((checked / total) * 100) : 0;
            if (percent > 100) percent = 100;

            $progressBar.val(percent);
            $progressText.text('Processing… ' + checked + ' / ' + total + ' (' + percent + '%)');

            if (resp.data.finished) {
                finishRun();
            } else {
                processBatch(resp.data.next_start);
            }
        }).fail(function() {
            running = false;
            setMessage('error', 'AJAX error while processing images.');
        });
    }

    function finishRun() {
        $.post(AIOSettings.ajax_url, {
            action: 'aio_finish_optimisation',
            nonce: AIOSettings.nonce,
            dry_run: dryRun ? 1 : 0
        }).done(function(resp) {
            running = false;

            if (!resp || !resp.success) {
                var msg = (resp && resp.data && resp.data.message) ? resp.data.message : 'Error finishing optimisation.';
                setMessage('error', msg);
                return;
            }

            var checkedFinal = parseInt(resp.data.checked, 10) || checked;
            var resizedFinal = parseInt(resp.data.resized, 10) || resized;
            var bytesFinal   = parseInt(resp.data.bytes_saved, 10) || bytesSaved;
            var bytesHuman   = resp.data.bytes_saved_human || humanBytes(bytesFinal);

            $progressBar.val(100);

            if (resp.data.dry_run) {
                $progressText.text('Dry run complete. Checked ' + checkedFinal + ' images. ' + resizedFinal + ' images would be updated.');
                setMessage(
                    'info',
                    'Dry run complete. Checked ' + checkedFinal + ' images. <strong>' + resizedFinal + '</strong> images would be resized and/or recompressed. No files were modified.'
                );
            } else {
                $progressText.text('Finished. Checked ' + checkedFinal + ' images, updated ' + resizedFinal + '. Saved approximately ' + bytesHuman + ' of disk space.');
                setMessage(
                    'success',
                    'Adam\'s Image Optimiser finished. Checked ' + checkedFinal + ' images, updated ' + resizedFinal + ', saved approximately ' + bytesHuman + ' of disk space. Reload the page to refresh the status panel.'
                );
            }
        }).fail(function() {
            running = false;
            setMessage('error', 'AJAX error while finishing optimisation.');
        });
    }

    if ($runButton.length) {
        $runButton.on('click', function(e) {
            e.preventDefault();
            if (running) {
                return;
            }
            prepareRun();
        });
    }
});
