<?php
/**
 * Plugin Name: Adam's Performance Manager
 * Description: Performance tweaks for WordPress – head cleanup, dashicons, Heartbeat control, query strings, WooCommerce bloat, DNS preconnect, and advanced optimisations.
 * Author: Adam / Dori Media
 * Version: 1.2.0
 * License: GPL2+
 * Text Domain: adams-performance-manager
 */

if (!defined('ABSPATH')) {
    exit;
}

class APM_Performance_Manager {

    const OPTION_KEY      = 'apm_performance_options';
    const CRON_DB_CLEANUP = 'apm_performance_db_cleanup';
    const OPTION_SPEED_TEST = 'apm_performance_speed_test';

    private static $instance = null;

    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // Runtime hooks
        add_action('init', array($this, 'init_cleanup'));
        add_action('wp_enqueue_scripts', array($this, 'maybe_disable_dashicons'), 100);
        add_action('wp_enqueue_scripts', array($this, 'maybe_trim_woocommerce_assets'), 99);

        add_filter('script_loader_src', array($this, 'maybe_remove_query_strings'), 15, 1);
        add_filter('style_loader_src', array($this, 'maybe_remove_query_strings'), 15, 1);

        add_filter('heartbeat_settings', array($this, 'filter_heartbeat_settings'));
        add_action('init', array($this, 'maybe_disable_heartbeat_frontend'), 1);

        add_filter('the_content', array($this, 'lazyload_iframes'), 20);

        add_action('wp_head', array($this, 'output_resource_hints'), 0);
        add_action('wp_head', array($this, 'output_critical_css'), 1);
        add_action('wp_head', array($this, 'maybe_preload_featured_image'), 2);

        add_filter('script_loader_tag', array($this, 'maybe_defer_scripts'), 10, 3);

        // Admin settings
        if (is_admin()) {
            add_action('admin_menu', array($this, 'register_settings_page'));
            add_action('admin_init', array($this, 'register_settings'));
            add_action('admin_init', array($this, 'handle_speed_test_request'));
        }

        // Cron: DB cleanup
        add_action(self::CRON_DB_CLEANUP, array($this, 'run_db_cleanup'));
    }

    public function activate() {
        $defaults = array(
            'cleanup_head'             => 1,
            'disable_emojis'           => 0, // often handled by security plugin
            'disable_dashicons'        => 1,
            'remove_query_strings'     => 1,
            'trim_woocommerce'         => 1,
            'heartbeat_frontend'       => 'disable', // disable / reduce / normal
            'heartbeat_admin'          => 'reduce',  // reduce / normal
            'heartbeat_admin_interval' => 60,        // seconds
            'dns_hosts'                => implode("\n", array(
                '//fonts.googleapis.com',
                '//fonts.gstatic.com',
                '//www.googletagmanager.com',
                '//www.google-analytics.com',
                '//static.cloudflareinsights.com',
            )),
            'lazyload_iframes'         => 1,

            // NEW: JS deferral / critical CSS / preload
            'defer_scripts'            => 0,
            'critical_css_global'      => '',
            'preload_featured_image'   => 0,

            // NEW: DB cleanup
            'db_cleanup_enabled'       => 1,
            'db_cleanup_delete_revisions' => 1,
            'db_cleanup_delete_trash'  => 1,
            'db_cleanup_delete_spam'   => 1,
            'db_cleanup_delete_transients' => 1,
            'db_cleanup_optimize_tables'   => 1,
        );

        $current = get_option(self::OPTION_KEY, array());
        update_option(self::OPTION_KEY, wp_parse_args($current, $defaults));

        // Schedule weekly DB cleanup
        if (!wp_next_scheduled(self::CRON_DB_CLEANUP)) {
            wp_schedule_event(time() + 3600, 'weekly', self::CRON_DB_CLEANUP);
        }
    }

    public function deactivate() {
        // Unschedule DB cleanup
        $timestamp = wp_next_scheduled(self::CRON_DB_CLEANUP);
        if ($timestamp) {
            wp_unschedule_event($timestamp, self::CRON_DB_CLEANUP);
        }
    }

    private function get_options() {
        $opts = get_option(self::OPTION_KEY, array());
        $defaults = array(
            'cleanup_head'             => 1,
            'disable_emojis'           => 0,
            'disable_dashicons'        => 1,
            'remove_query_strings'     => 1,
            'trim_woocommerce'         => 1,
            'heartbeat_frontend'       => 'disable',
            'heartbeat_admin'          => 'reduce',
            'heartbeat_admin_interval' => 60,
            'dns_hosts'                => '',
            'lazyload_iframes'         => 1,
            'defer_scripts'            => 0,
            'critical_css_global'      => '',
            'preload_featured_image'   => 0,
            'db_cleanup_enabled'       => 1,
            'db_cleanup_delete_revisions' => 1,
            'db_cleanup_delete_trash'  => 1,
            'db_cleanup_delete_spam'   => 1,
            'db_cleanup_delete_transients' => 1,
            'db_cleanup_optimize_tables'   => 1,
        );
        return wp_parse_args($opts, $defaults);
    }

    /* ========================================================
     * Core cleanup
     * ===================================================== */

    public function init_cleanup() {
        $opts = $this->get_options();

        if (!empty($opts['cleanup_head'])) {
            // Remove various cruft from <head>
            remove_action('wp_head', 'rsd_link');
            remove_action('wp_head', 'wlwmanifest_link');
            remove_action('wp_head', 'wp_shortlink_wp_head', 10);
            remove_action('wp_head', 'rest_output_link_wp_head', 10);
            remove_action('wp_head', 'wp_oembed_add_discovery_links', 10);
            remove_action('wp_head', 'adjacent_posts_rel_link_wp_head', 10, 0);
        }

        if (!empty($opts['disable_emojis'])) {
            remove_action('wp_head', 'print_emoji_detection_script', 7);
            remove_action('admin_print_scripts', 'print_emoji_detection_script');
            remove_action('wp_print_styles', 'print_emoji_styles');
            remove_action('admin_print_styles', 'print_emoji_styles');
            remove_filter('the_content_feed', 'wp_staticize_emoji');
            remove_filter('comment_text_rss', 'wp_staticize_emoji');
            remove_filter('wp_mail', 'wp_staticize_emoji_for_email');
        }
    }

    /* ========================================================
     * Dashicons
     * ===================================================== */

    public function maybe_disable_dashicons() {
        $opts = $this->get_options();
        if (empty($opts['disable_dashicons'])) {
            return;
        }

        if (!is_user_logged_in() && !is_admin()) {
            wp_deregister_style('dashicons');
        }
    }

    /* ========================================================
     * WooCommerce trimming
     * ===================================================== */

    public function maybe_trim_woocommerce_assets() {
        $opts = $this->get_options();
        if (empty($opts['trim_woocommerce'])) {
            return;
        }

        if (!defined('WC_VERSION')) {
            return;
        }

        if (function_exists('is_woocommerce')) {
            if (is_woocommerce() || is_cart() || is_checkout() || is_account_page()) {
                return; // keep assets on shop pages
            }
        }

        // Dequeue some heavy WC assets on non-shop pages
        wp_dequeue_style('woocommerce-general');
        wp_dequeue_style('woocommerce-layout');
        wp_dequeue_style('woocommerce-smallscreen');
        wp_dequeue_script('wc-add-to-cart');
        wp_dequeue_script('wc-cart-fragments');
    }

    /* ========================================================
     * Query strings
     * ===================================================== */

    public function maybe_remove_query_strings($src) {
        $opts = $this->get_options();
        if (empty($opts['remove_query_strings'])) {
            return $src;
        }

        // Only strip ?ver= query to avoid breaking other params
        $parts = explode('?ver=', $src);
        return $parts[0];
    }

    /* ========================================================
     * Heartbeat
     * ===================================================== */

    public function maybe_disable_heartbeat_frontend() {
        $opts = $this->get_options();

        // Disable heartbeat on front-end if requested
        if (!is_admin() && $opts['heartbeat_frontend'] === 'disable') {
            wp_deregister_script('heartbeat');
        }
    }

    public function filter_heartbeat_settings($settings) {
        $opts = $this->get_options();

        if (is_admin()) {
            if ($opts['heartbeat_admin'] === 'reduce') {
                $interval = max(15, (int) $opts['heartbeat_admin_interval']);
                $settings['interval'] = $interval;
            }
        } else {
            if ($opts['heartbeat_frontend'] === 'reduce') {
                $settings['interval'] = 60;
            }
        }

        return $settings;
    }

    /* ========================================================
     * Lazyload iframes
     * ===================================================== */

    public function lazyload_iframes($content) {
        $opts = $this->get_options();
        if (empty($opts['lazyload_iframes'])) {
            return $content;
        }

        if (is_admin()) {
            return $content;
        }

        // Add loading="lazy" to iframes that don't have it
        $content = preg_replace_callback(
            '#<iframe([^>]+)>#i',
            function ($matches) {
                $tag = $matches[0];
                // skip if already has loading attribute
                if (stripos($tag, 'loading=') !== false) {
                    return $tag;
                }
                // inject loading="lazy"
                $tag = str_replace('<iframe', '<iframe loading="lazy"', $tag);
                return $tag;
            },
            $content
        );

        return $content;
    }

    /* ========================================================
     * DNS Prefetch / Preconnect
     * ===================================================== */

    public function output_resource_hints() {
        $opts = $this->get_options();
        $hosts_raw = trim($opts['dns_hosts']);
        if (empty($hosts_raw)) {
            return;
        }

        $hosts = array_filter(array_map('trim', explode("\n", $hosts_raw)));
        if (empty($hosts)) {
            return;
        }

        foreach ($hosts as $host) {
            // Simple normalisation – allow //host form
            $url = $host;
            if (strpos($url, '//') === 0) {
                $url = 'https:' . $url;
            }
            $url = esc_url($url);
            if (!$url) {
                continue;
            }
            echo '<link rel="preconnect" href="' . esc_url($url) . '" crossorigin>' . "\n";
            echo '<link rel="dns-prefetch" href="' . esc_url($url) . '">' . "\n";
        }
    }

    /* ========================================================
     * Critical CSS + Featured Image Preload
     * ===================================================== */

    public function output_critical_css() {
        $opts = $this->get_options();
        $css  = trim($opts['critical_css_global']);
        if (empty($css)) {
            return;
        }

        if (is_admin()) {
            return;
        }

        echo "<style id=\"apm-critical-css\">\n" . $css . "\n</style>\n";
    }

    public function maybe_preload_featured_image() {
        $opts = $this->get_options();
        if (empty($opts['preload_featured_image'])) {
            return;
        }

        if (!is_singular() || is_admin()) {
            return;
        }

        if (!has_post_thumbnail()) {
            return;
        }

        $img = wp_get_attachment_image_src(get_post_thumbnail_id(), 'full');
        if (!$img || empty($img[0])) {
            return;
        }

        $url = esc_url($img[0]);
        echo '<link rel="preload" as="image" href="' . $url . '">' . "\n";
    }

    /* ========================================================
     * JS deferral
     * ===================================================== */

    public function maybe_defer_scripts($tag, $handle, $src) {
        $opts = $this->get_options();
        if (empty($opts['defer_scripts'])) {
            return $tag;
        }

        if (is_admin()) {
            return $tag;
        }

        // Don't defer critical/core scripts
        $no_defer_handles = array(
            'jquery',
            'jquery-core',
            'jquery-migrate',
            'wp-util',
            'wp-i18n',
            'wp-hooks',
            'elementor-frontend',
            'elementor-pro-frontend',
            'elementor-waypoints',
            'elementor-admin-bar',
            'cf-turnstile',
        );

        if (in_array($handle, $no_defer_handles, true)) {
            return $tag;
        }

        // Already deferred or async?
        if (strpos($tag, ' defer') !== false || strpos($tag, ' async') !== false) {
            return $tag;
        }

        // Only apply to local and https scripts; skip if weird
        if (!empty($src) && strpos($src, 'http') === 0) {
            // add defer attribute
            $tag = str_replace('<script ', '<script defer ', $tag);
        }

        return $tag;
    }

    /* ========================================================
     * DB Cleanup (weekly cron)
     * ===================================================== */

    public function run_db_cleanup() {
        global $wpdb;
        $opts = $this->get_options();

        if (empty($opts['db_cleanup_enabled'])) {
            return;
        }

        // Delete post revisions
        if (!empty($opts['db_cleanup_delete_revisions'])) {
            $wpdb->query(
                "DELETE FROM {$wpdb->posts}
                 WHERE post_type = 'revision'"
            );
        }

        // Delete trashed posts & pages
        if (!empty($opts['db_cleanup_delete_trash'])) {
            $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$wpdb->posts}
                     WHERE post_status = %s",
                    'trash'
                )
            );
        }

        // Delete spam / trash comments
        if (!empty($opts['db_cleanup_delete_spam'])) {
            $wpdb->query(
                "DELETE FROM {$wpdb->comments}
                 WHERE comment_approved = 'spam' OR comment_approved = 'trash'"
            );
        }

        // Delete expired transients
        if (!empty($opts['db_cleanup_delete_transients'])) {
            if (function_exists('delete_expired_transients')) {
                delete_expired_transients();
            } else {
                // Fallback: remove *_transient_*_timeout entries past now
                $time = time();
                $wpdb->query(
                    $wpdb->prepare(
                        "DELETE FROM {$wpdb->options}
                         WHERE option_name LIKE %s
                         AND option_value < %d",
                        $wpdb->esc_like('_transient_timeout_') . '%',
                        $time
                    )
                );
            }
        }

        // Optimize main tables
        if (!empty($opts['db_cleanup_optimize_tables'])) {
            $tables = array(
                $wpdb->posts,
                $wpdb->postmeta,
                $wpdb->options,
                $wpdb->comments,
                $wpdb->commentmeta,
                $wpdb->terms,
                $wpdb->term_taxonomy,
                $wpdb->term_relationships,
                $wpdb->users,
                $wpdb->usermeta,
            );

            foreach ($tables as $table) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                $wpdb->query("OPTIMIZE TABLE {$table}");
            }
        }
    }

    /* ========================================================
     * Settings
     * ===================================================== */

    public function register_settings_page() {
        add_options_page(
            "Adam's Performance Manager",
            'Performance Manager',
            'manage_options',
            'apm-performance-manager',
            array($this, 'render_settings_page')
        );
    }

    public function register_settings() {
        register_setting(
            'apm_performance_group',
            self::OPTION_KEY,
            array($this, 'sanitize_options')
        );

        add_settings_section(
            'apm_main',
            'Adam’s Performance Manager – Core',
            function () {
                echo '<p>Baseline performance tweaks for this WordPress site.</p>';
            },
            'apm-performance-manager'
        );

        // Simple checkboxes – core
        $checkbox_fields = array(
            'cleanup_head'         => 'Clean up &lt;head&gt; (remove RSD, WLW, shortlink, extra REST/oEmbed links)',
            'disable_emojis'       => 'Disable emojis (emoji scripts/styles)',
            'disable_dashicons'    => 'Disable Dashicons on front-end for visitors',
            'remove_query_strings' => 'Remove ?ver= query strings from CSS/JS',
            'trim_woocommerce'     => 'Trim WooCommerce assets on non-shop pages',
            'lazyload_iframes'     => 'Lazy-load iframes in content',
        );

        foreach ($checkbox_fields as $key => $label) {
            add_settings_field(
                $key,
                $label,
                array($this, 'render_checkbox_field'),
                'apm-performance-manager',
                'apm_main',
                array('key' => $key)
            );
        }

        // Heartbeat options
        add_settings_field(
            'heartbeat_frontend',
            'Heartbeat (front-end)',
            array($this, 'field_heartbeat_frontend'),
            'apm-performance-manager',
            'apm_main'
        );

        add_settings_field(
            'heartbeat_admin',
            'Heartbeat (admin)',
            array($this, 'field_heartbeat_admin'),
            'apm-performance-manager',
            'apm_main'
        );

        // DNS hosts
        add_settings_field(
            'dns_hosts',
            'DNS prefetch / preconnect hosts',
            array($this, 'field_dns_hosts'),
            'apm-performance-manager',
            'apm_main'
        );

        // Advanced section
        add_settings_section(
            'apm_advanced',
            'Advanced Optimisations',
            function () {
                echo '<p>Extra tuning for JS, critical CSS and media. Leave defaults if unsure.</p>';
            },
            'apm-performance-manager'
        );

        add_settings_field(
            'defer_scripts',
            'Defer non-critical scripts',
            array($this, 'render_checkbox_field'),
            'apm-performance-manager',
            'apm_advanced',
            array('key' => 'defer_scripts')
        );

        add_settings_field(
            'critical_css_global',
            'Global Critical CSS',
            array($this, 'field_critical_css'),
            'apm-performance-manager',
            'apm_advanced'
        );

        add_settings_field(
            'preload_featured_image',
            'Preload featured image on single posts/pages',
            array($this, 'render_checkbox_field'),
            'apm-performance-manager',
            'apm_advanced',
            array('key' => 'preload_featured_image')
        );

        // DB Cleanup section
        add_settings_section(
            'apm_db',
            'Database Cleanup & Maintenance',
            function () {
                echo '<p>Weekly cleanup job to reduce DB bloat. Runs via WP-Cron.</p>';
            },
            'apm-performance-manager'
        );

        add_settings_field(
            'db_cleanup_enabled',
            'Enable weekly DB cleanup',
            array($this, 'render_checkbox_field'),
            'apm-performance-manager',
            'apm_db',
            array('key' => 'db_cleanup_enabled')
        );

        $db_fields = array(
            'db_cleanup_delete_revisions'   => 'Delete all post revisions',
            'db_cleanup_delete_trash'       => 'Delete trashed posts/pages',
            'db_cleanup_delete_spam'        => 'Delete spam/trash comments',
            'db_cleanup_delete_transients'  => 'Delete expired transients',
            'db_cleanup_optimize_tables'    => 'Optimize main WP tables',
        );

        foreach ($db_fields as $key => $label) {
            add_settings_field(
                $key,
                $label,
                array($this, 'render_checkbox_field'),
                'apm-performance-manager',
                'apm_db',
                array('key' => $key)
            );
        }
    }

    public function sanitize_options($input) {
        $output = $this->get_options();

        $bool_keys = array(
            'cleanup_head',
            'disable_emojis',
            'disable_dashicons',
            'remove_query_strings',
            'trim_woocommerce',
            'lazyload_iframes',
            'defer_scripts',
            'preload_featured_image',
            'db_cleanup_enabled',
            'db_cleanup_delete_revisions',
            'db_cleanup_delete_trash',
            'db_cleanup_delete_spam',
            'db_cleanup_delete_transients',
            'db_cleanup_optimize_tables',
        );

        foreach ($bool_keys as $key) {
            $output[$key] = !empty($input[$key]) ? 1 : 0;
        }

        if (isset($input['heartbeat_frontend'])) {
            $val = $input['heartbeat_frontend'];
            if (!in_array($val, array('disable', 'reduce', 'normal'), true)) {
                $val = 'disable';
            }
            $output['heartbeat_frontend'] = $val;
        }

        if (isset($input['heartbeat_admin'])) {
            $val = $input['heartbeat_admin'];
            if (!in_array($val, array('reduce', 'normal'), true)) {
                $val = 'reduce';
            }
            $output['heartbeat_admin'] = $val;
        }

        if (isset($input['heartbeat_admin_interval'])) {
            $output['heartbeat_admin_interval'] = max(15, (int) $input['heartbeat_admin_interval']);
        }

        if (isset($input['dns_hosts'])) {
            $output['dns_hosts'] = trim(wp_kses_post($input['dns_hosts']));
        }

        if (isset($input['critical_css_global'])) {
            $output['critical_css_global'] = trim(wp_kses_post($input['critical_css_global']));
        }

        return $output;
    }

    public function render_checkbox_field($args) {
        $key  = $args['key'];
        $opts = $this->get_options();
        ?>
        <label>
            <input type="checkbox"
                   name="<?php echo esc_attr(self::OPTION_KEY . '[' . $key . ']'); ?>"
                   value="1" <?php checked(!empty($opts[$key]), 1); ?>>
        </label>
        <?php
    }

    public function field_heartbeat_frontend() {
        $opts = $this->get_options();
        $val  = $opts['heartbeat_frontend'];
        ?>
        <select name="<?php echo esc_attr(self::OPTION_KEY); ?>[heartbeat_frontend]">
            <option value="disable" <?php selected($val, 'disable'); ?>>Disable on front-end (recommended)</option>
            <option value="reduce"  <?php selected($val, 'reduce');  ?>>Reduce frequency</option>
            <option value="normal"  <?php selected($val, 'normal');  ?>>Leave default</option>
        </select>
        <p class="description">Heartbeat controls autosave and some AJAX stuff. Disabling it on the front-end is usually safe.</p>
        <?php
    }

    public function field_heartbeat_admin() {
        $opts = $this->get_options();
        $val  = $opts['heartbeat_admin'];
        ?>
        <select name="<?php echo esc_attr(self::OPTION_KEY); ?>[heartbeat_admin]">
            <option value="reduce" <?php selected($val, 'reduce'); ?>>Reduce frequency (recommended)</option>
            <option value="normal" <?php selected($val, 'normal'); ?>>Leave default</option>
        </select>
        <p class="description">
            Interval (seconds):
            <input type="number"
                   name="<?php echo esc_attr(self::OPTION_KEY); ?>[heartbeat_admin_interval]"
                   value="<?php echo (int) $opts['heartbeat_admin_interval']; ?>"
                   min="15"
                   style="width:80px;">
        </p>
        <?php
    }

    public function field_dns_hosts() {
        $opts = $this->get_options();
        ?>
        <textarea
            name="<?php echo esc_attr(self::OPTION_KEY); ?>[dns_hosts]"
            rows="5"
            cols="60"
        ><?php echo esc_textarea($opts['dns_hosts']); ?></textarea>
        <p class="description">
            One host per line. You can use <code>//example.com</code> or <code>https://example.com</code>.  
            Preconnect + DNS prefetch hints help with Google Fonts, GA, Cloudflare, CDNs, etc.
        </p>
        <?php
    }

    public function field_critical_css() {
        $opts = $this->get_options();
        ?>
        <textarea
            name="<?php echo esc_attr(self::OPTION_KEY); ?>[critical_css_global]"
            rows="6"
            cols="80"
        ><?php echo esc_textarea($opts['critical_css_global']); ?></textarea>
        <p class="description">
            Paste critical CSS here (above-the-fold). It will be inlined into &lt;head&gt; on the front-end.
            Leave empty if you don’t want to use critical CSS yet.
        </p>
        <?php
    }

    
    /**
     * Handle POST from the speed test forms on the settings page.
     */
    public function handle_speed_test_request() {
        if (!isset($_POST['apm_speed_test_action'])) {
            return;
        }

        if (!current_user_can('manage_options')) {
            return;
        }

        if (!isset($_POST['apm_speed_test_nonce']) || !wp_verify_nonce($_POST['apm_speed_test_nonce'], 'apm_speed_test')) {
            return;
        }

        $label = ($_POST['apm_speed_test_action'] === 'after') ? 'after' : 'before';

        $raw_url = isset($_POST['apm_speed_test_url']) ? trim(wp_unslash($_POST['apm_speed_test_url'])) : '';
        $url     = esc_url_raw($raw_url);
        if (empty($url)) {
            $url = home_url('/');
        }

        $result = $this->run_speed_test_for_url($url);

        if ($result && empty($result['error'])) {
            $stored = get_option(self::OPTION_SPEED_TEST, array(
                'before' => null,
                'after'  => null,
            ));

            $stored[$label] = $result;
            update_option(self::OPTION_SPEED_TEST, $stored);

            set_transient(
                'apm_speed_test_notice',
                array(
                    'type'    => 'success',
                    'message' => sprintf(
                        /* translators: 1: label (Before/After), 2: time in ms */
                        __('%1$s speed test completed in %2$d ms (avg).', 'adams-performance-manager'),
                        ucfirst($label),
                        isset($result['time_ms']) ? (int) $result['time_ms'] : 0
                    ),
                ),
                30
            );
        } else {
            $error_message = isset($result['error']) ? $result['error'] : __('Speed test failed.', 'adams-performance-manager');
            set_transient(
                'apm_speed_test_notice',
                array(
                    'type'    => 'error',
                    'message' => $error_message,
                ),
                30
            );
        }

        // Redirect back to settings page.
        wp_safe_redirect(add_query_arg(array('page' => 'apm-performance-manager'), admin_url('options-general.php')));
        exit;
    }

    /**
     * Run a small series of HTTP requests against a URL and average the time.
     *
     * @param string $url
     * @return array
     */
    protected function run_speed_test_for_url($url) {
        $runs       = 3;
        $times      = array();
        $status     = null;
        $body_size  = null;

        for ($i = 0; $i < $runs; $i++) {
            $start = microtime(true);

            $response = wp_remote_get($url, array(
                'timeout'     => 20,
                'redirection' => 5,
                'sslverify'   => false,
                'headers'     => array(
                    'Cache-Control' => 'no-cache',
                    'Pragma'        => 'no-cache',
                ),
            ));

            $end = microtime(true);

            if (is_wp_error($response)) {
                return array(
                    'error' => sprintf(
                        /* translators: %s: error message */
                        __('Error during speed test: %s', 'adams-performance-manager'),
                        $response->get_error_message()
                    ),
                );
            }

            $times[]    = ($end - $start) * 1000; // ms
            $status     = wp_remote_retrieve_response_code($response);
            $body       = wp_remote_retrieve_body($response);
            $body_size  = strlen($body); // bytes
        }

        if (empty($times)) {
            return array(
                'error' => __('No timings recorded for speed test.', 'adams-performance-manager'),
            );
        }

        $avg_time = (int) round(array_sum($times) / count($times));
        $kb_size  = $body_size ? round($body_size / 1024, 1) : 0;

        return array(
            'url'       => esc_url_raw($url),
            'time_ms'   => $avg_time,
            'status'    => (int) $status,
            'size_kb'   => $kb_size,
            'runs'      => $runs,
            'tested_at' => current_time('mysql'),
        );
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Fetch any speed test notice.
        $notice = get_transient('apm_speed_test_notice');
        if ($notice) {
            delete_transient('apm_speed_test_notice');
        }

        $speed_results = get_option(self::OPTION_SPEED_TEST, array(
            'before' => null,
            'after'  => null,
        ));

        $before = isset($speed_results['before']) ? $speed_results['before'] : null;
        $after  = isset($speed_results['after']) ? $speed_results['after'] : null;

        $improvement = '';
        if ($before && $after && !empty($before['time_ms'])) {
            $diff    = (int) $before['time_ms'] - (int) $after['time_ms'];
            $percent = ($diff / (int) $before['time_ms']) * 100;
            $improvement = sprintf(
                /* translators: 1: time difference in ms, 2: percentage */
                __('Improvement: %1$d ms faster (%.1f%%)', 'adams-performance-manager'),
                $diff,
                $percent
            );
        }
        ?>
        <div class="wrap">
            <h1>Adam's Performance Manager</h1>
            <p><?php esc_html_e('Baseline performance tuning for this WordPress site.', 'adams-performance-manager'); ?></p>

            <?php if (!empty($notice)) : ?>
                <?php
                $class = ($notice['type'] === 'success') ? 'notice notice-success' : 'notice notice-error';
                ?>
                <div class="<?php echo esc_attr($class); ?>">
                    <p><?php echo esc_html($notice['message']); ?></p>
                </div>
            <?php endif; ?>

            <form method="post" action="options.php">
                <?php
                settings_fields('apm_performance_group');
                do_settings_sections('apm-performance-manager');
                submit_button();
                ?>
            </form>

            <hr />

            <h2><?php esc_html_e('Speed Test (Before / After)', 'adams-performance-manager'); ?></h2>
            <p><?php esc_html_e('Run a quick before/after speed test of a key URL to see how much the performance tweaks help.', 'adams-performance-manager'); ?></p>

            <form method="post">
                <?php wp_nonce_field('apm_speed_test', 'apm_speed_test_nonce'); ?>

                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="apm_speed_test_url"><?php esc_html_e('Test URL', 'adams-performance-manager'); ?></label>
                        </th>
                        <td>
                            <input type="url"
                                   id="apm_speed_test_url"
                                   name="apm_speed_test_url"
                                   class="regular-text"
                                   value="<?php echo esc_attr($before && !empty($before['url']) ? $before['url'] : home_url('/')); ?>" />
                            <p class="description">
                                <?php esc_html_e('Usually your homepage. You can change this to a key landing page if you prefer.', 'adams-performance-manager'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <p>
                    <button type="submit" name="apm_speed_test_action" value="before" class="button">
                        <?php esc_html_e('Run baseline test (Before)', 'adams-performance-manager'); ?>
                    </button>
                    <button type="submit" name="apm_speed_test_action" value="after" class="button button-primary">
                        <?php esc_html_e('Run test (After optimisations)', 'adams-performance-manager'); ?>
                    </button>
                </p>
            </form>

            <?php if ($before || $after) : ?>
                <h3><?php esc_html_e('Results', 'adams-performance-manager'); ?></h3>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Label', 'adams-performance-manager'); ?></th>
                            <th><?php esc_html_e('URL', 'adams-performance-manager'); ?></th>
                            <th><?php esc_html_e('Avg. Time (ms)', 'adams-performance-manager'); ?></th>
                            <th><?php esc_html_e('Status', 'adams-performance-manager'); ?></th>
                            <th><?php esc_html_e('HTML Size (KB)', 'adams-performance-manager'); ?></th>
                            <th><?php esc_html_e('Runs', 'adams-performance-manager'); ?></th>
                            <th><?php esc_html_e('Tested At', 'adams-performance-manager'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ($before) : ?>
                            <tr>
                                <td><?php esc_html_e('Before', 'adams-performance-manager'); ?></td>
                                <td><?php echo esc_html($before['url']); ?></td>
                                <td><?php echo isset($before['time_ms']) ? (int) $before['time_ms'] : ''; ?></td>
                                <td><?php echo isset($before['status']) ? (int) $before['status'] : ''; ?></td>
                                <td><?php echo isset($before['size_kb']) ? esc_html($before['size_kb']) : ''; ?></td>
                                <td><?php echo isset($before['runs']) ? (int) $before['runs'] : ''; ?></td>
                                <td><?php echo isset($before['tested_at']) ? esc_html($before['tested_at']) : ''; ?></td>
                            </tr>
                        <?php endif; ?>
                        <?php if ($after) : ?>
                            <tr>
                                <td><?php esc_html_e('After', 'adams-performance-manager'); ?></td>
                                <td><?php echo esc_html($after['url']); ?></td>
                                <td><?php echo isset($after['time_ms']) ? (int) $after['time_ms'] : ''; ?></td>
                                <td><?php echo isset($after['status']) ? (int) $after['status'] : ''; ?></td>
                                <td><?php echo isset($after['size_kb']) ? esc_html($after['size_kb']) : ''; ?></td>
                                <td><?php echo isset($after['runs']) ? (int) $after['runs'] : ''; ?></td>
                                <td><?php echo isset($after['tested_at']) ? esc_html($after['tested_at']) : ''; ?></td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>

                <?php if (!empty($improvement)) : ?>
                    <p><strong><?php echo esc_html($improvement); ?></strong></p>
                <?php endif; ?>
            <?php endif; ?>

            <hr />
            <p>
                <em><?php esc_html_e('Tip: For full page caching, still use Cloudflare APO or your host’s page cache. Adam’s Performance Manager focuses on safe, code-level optimisations that work across all sites.', 'adams-performance-manager'); ?></em>
            </p>
        </div>
        <?php
    }
}

APM_Performance_Manager::get_instance();