<?php
/**
 * Plugin Name: Adam's Plugin Hub
 * Description: A simple hub inside WordPress that lets you browse and install custom plugins from your own private catalog.
 * Version:     0.1.3
 * Author:      Adam
 * Text Domain: adams-plugin-hub
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'AdamsPluginHub' ) ) :

class AdamsPluginHub {

    const VERSION     = '0.1.3';
    // Point this at your live catalog endpoint.
    const CATALOG_URL = 'https://plugins.dorimedia.co.nz/api/wp-plugin-catalog.json';

    public function __construct() {
        add_action( 'admin_menu', [ $this, 'register_admin_menu' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'admin_assets' ] );
        add_action( 'rest_api_init', [ $this, 'register_rest_routes' ] );
    }

    public function register_admin_menu() {
        add_menu_page(
            __( "Plugin Hub", 'adams-plugin-hub' ),
            __( "Plugin Hub", 'adams-plugin-hub' ),
            'activate_plugins', // lowered from install_plugins
            'adams-plugin-hub',
            [ $this, 'render_page' ],
            'dashicons-admin-plugins',
            59
        );
    }

    public function admin_assets( $hook ) {
        if ( $hook !== 'toplevel_page_adams-plugin-hub' ) {
            return;
        }

        wp_enqueue_style(
            'adams-plugin-hub-admin',
            plugin_dir_url( __FILE__ ) . 'assets/css/admin.css',
            [],
            self::VERSION
        );

        wp_enqueue_script(
            'adams-plugin-hub-admin',
            plugin_dir_url( __FILE__ ) . 'assets/js/admin.js',
            [ 'jquery', 'wp-util' ],
            self::VERSION,
            true
        );

        // Installed and active plugin info for the UI.
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $all_plugins    = get_plugins();
        $active_plugins = get_option( 'active_plugins', [] );

        $installed = [];
        foreach ( $all_plugins as $file => $data ) {
            $slug = ! empty( $data['TextDomain'] ) ? $data['TextDomain'] : dirname( $file );

            $installed[ $slug ] = [
                'file'    => $file,
                'version' => isset( $data['Version'] ) ? $data['Version'] : '',
                'active'  => in_array( $file, $active_plugins, true ),
            ];
        }

        wp_localize_script(
            'adams-plugin-hub-admin',
            'AdamsPluginHubConfig',
            [
                'restUrl'   => esc_url_raw( rest_url( 'adams-plugin-hub/v1' ) ),
                'nonce'     => wp_create_nonce( 'wp_rest' ),
                'installed' => $installed,
            ]
        );
    }

    public function register_rest_routes() {
        register_rest_route(
            'adams-plugin-hub/v1',
            '/catalog',
            [
                'methods'             => 'GET',
                'callback'            => [ $this, 'rest_get_catalog' ],
                'permission_callback' => function () {
                    return current_user_can( 'activate_plugins' );
                },
            ]
        );

        register_rest_route(
            'adams-plugin-hub/v1',
            '/install',
            [
                'methods'             => 'POST',
                'callback'            => [ $this, 'rest_install_plugin' ],
                'permission_callback' => function () {
                    return current_user_can( 'activate_plugins' );
                },
            ]
        );

        register_rest_route(
            'adams-plugin-hub/v1',
            '/toggle',
            [
                'methods'             => 'POST',
                'callback'            => [ $this, 'rest_toggle_plugin' ],
                'permission_callback' => function () {
                    return current_user_can( 'activate_plugins' );
                },
            ]
        );
    }

    /**
     * Fetch plugin catalog from your central API and return it to JS.
     */
    public function rest_get_catalog( WP_REST_Request $request ) {
        $response = wp_remote_get( self::CATALOG_URL, [
            'timeout' => 10,
        ] );

        if ( is_wp_error( $response ) ) {
            return new WP_REST_Response(
                [
                    'success' => false,
                    'message' => $response->get_error_message(),
                    'plugins' => [],
                ],
                500
            );
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( 200 !== $code ) {
            return new WP_REST_Response(
                [
                    'success' => false,
                    'message' => sprintf( 'Unexpected response code: %d', $code ),
                    'plugins' => [],
                ],
                500
            );
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( ! is_array( $data ) ) {
            return new WP_REST_Response(
                [
                    'success' => false,
                    'message' => 'Invalid catalog format.',
                    'plugins' => [],
                ],
                500
            );
        }

        return [
            'success' => true,
            'plugins' => $data,
        ];
    }

    /**
     * Helper: find plugin file by catalog slug, matching TextDomain first.
     */
    protected function find_plugin_file_by_slug( $slug ) {
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $all_plugins = get_plugins();

        foreach ( $all_plugins as $file => $data ) {
            $candidate = ! empty( $data['TextDomain'] ) ? $data['TextDomain'] : dirname( $file );
            if ( $candidate === $slug ) {
                return $file;
            }
        }

        return '';
    }

    /**
     * Install (and activate) a plugin from the hub.
     */
    public function rest_install_plugin( WP_REST_Request $request ) {
        $slug         = sanitize_text_field( $request->get_param( 'slug' ) );
        $download_url = esc_url_raw( $request->get_param( 'download_url' ) );

        if ( empty( $slug ) || empty( $download_url ) ) {
            return new WP_REST_Response(
                [
                    'success' => false,
                    'message' => 'Missing slug or download_url.',
                ],
                400
            );
        }

        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

        $url  = esc_url_raw( admin_url( 'admin.php?page=adams-plugin-hub' ) );

        $creds = request_filesystem_credentials( $url, '', false, false, [] );
        if ( false === $creds ) {
            return new WP_REST_Response(
                [
                    'success' => false,
                    'message' => 'Could not get filesystem credentials.',
                ],
                500
            );
        }

        if ( ! WP_Filesystem( $creds ) ) {
            return new WP_REST_Response(
                [
                    'success' => false,
                    'message' => 'Filesystem initialisation failed.',
                ],
                500
            );
        }

        $upgrader = new Plugin_Upgrader( new Automatic_Upgrader_Skin() );
        $result   = $upgrader->install( $download_url );

        if ( is_wp_error( $result ) ) {
            return new WP_REST_Response(
                [
                    'success' => false,
                    'message' => $result->get_error_message(),
                ],
                500
            );
        }

        // Try to find and activate the plugin by slug/TextDomain.
        $plugin_file = $this->find_plugin_file_by_slug( $slug );

        if ( $plugin_file ) {
            $activate_result = activate_plugin( $plugin_file );
            if ( is_wp_error( $activate_result ) ) {
                return new WP_REST_Response(
                    [
                        'success' => false,
                        'message' => $activate_result->get_error_message(),
                    ],
                    500
                );
            }
        }

        return [
            'success' => true,
            'status'  => 'installed_and_activated',
        ];
    }

    /**
     * Toggle plugin activation status.
     */
    public function rest_toggle_plugin( WP_REST_Request $request ) {
        $slug   = sanitize_text_field( $request->get_param( 'slug' ) );
        $action = sanitize_text_field( $request->get_param( 'action' ) ); // 'activate' or 'deactivate'

        if ( empty( $slug ) || empty( $action ) ) {
            return new WP_REST_Response(
                [
                    'success' => false,
                    'message' => 'Missing slug or action.',
                ],
                400
            );
        }

        if ( ! function_exists( 'is_plugin_active' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugin_file = $this->find_plugin_file_by_slug( $slug );

        if ( ! $plugin_file ) {
            return new WP_REST_Response(
                [
                    'success' => false,
                    'message' => 'Plugin file not found for slug ' . $slug,
                ],
                404
            );
        }

        if ( 'activate' === $action ) {
            $result = activate_plugin( $plugin_file );
            if ( is_wp_error( $result ) ) {
                return new WP_REST_Response(
                    [
                        'success' => false,
                        'message' => $result->get_error_message(),
                    ],
                    500
                );
            }
            return [
                'success' => true,
                'status'  => 'activated',
            ];
        }

        if ( 'deactivate' === $action ) {
            deactivate_plugins( $plugin_file, false, false );
            return [
                'success' => true,
                'status'  => 'deactivated',
            ];
        }

        return new WP_REST_Response(
            [
                'success' => false,
                'message' => 'Unknown action.',
            ],
            400
        );
    }

    public function render_page() {
        ?>
        <div class="wrap adams-plugin-hub">
            <h1><?php esc_html_e( "Adam's Plugin Hub", 'adams-plugin-hub' ); ?></h1>
            <p class="adams-plugin-hub-subtitle">
                <?php esc_html_e( 'Browse and install plugins from your private catalog. Toggle them on/off with a single click.', 'adams-plugin-hub' ); ?>
            </p>

            <div id="adams-plugin-hub-root">
                <div class="adams-plugin-hub-loading">
                    <?php esc_html_e( 'Loading plugin catalog…', 'adams-plugin-hub' ); ?>
                </div>
            </div>
        </div>
        <?php
    }
}

new AdamsPluginHub();

endif;
