jQuery(function($) {
    var cfg = window.AdamsPluginHubConfig || {};
    var installed = cfg.installed || {};

    function compareVersions(a, b) {
        if (!a || !b) return 0;
        var aParts = String(a).split('.');
        var bParts = String(b).split('.');
        var len = Math.max(aParts.length, bParts.length);

        for (var i = 0; i < len; i++) {
            var aNum = parseInt(aParts[i] || '0', 10);
            var bNum = parseInt(bParts[i] || '0', 10);

            if (aNum > bNum) return 1;
            if (aNum < bNum) return -1;
        }
        return 0;
    }

    function renderPlugins(plugins) {
        var $root = $('#adams-plugin-hub-root');
        $root.empty();

        if (!plugins.length) {
            $root.append(
                $('<p/>').text('No plugins found in the catalog. Check your catalog URL.')
            );
            return;
        }

        var $grid = $('<div class="adams-plugin-hub-grid"/>');

        plugins.forEach(function(plugin) {
            var slug = plugin.slug;
            var local = installed[slug] || null;
            var isInstalled = !!local;
            var isActive = local && local.active;
            var installedVersion = local && local.version ? local.version : null;
            var latestVersion = plugin.version || null;
            var hasUpdate = false;

            if (isInstalled && installedVersion && latestVersion) {
                // installed < latest → update available
                hasUpdate = compareVersions(installedVersion, latestVersion) === -1;
            }

            var $card = $('<div class="adams-plugin-hub-card"/>');

            var $header = $('<div class="adams-plugin-hub-card-header"/>');
            var $title = $('<h2/>').text(plugin.name || slug);
            var $version = $('<span class="adams-plugin-hub-version"/>')
                .text('v' + (latestVersion || ''));
            $header.append($title).append($version);

            var $desc = $('<p class="adams-plugin-hub-desc"/>')
                .text(plugin.description || '');

            var $status = $('<div class="adams-plugin-hub-status"/>');

            if (isInstalled) {
                if (hasUpdate) {
                    $status.append(
                        $('<span class="adams-plugin-hub-badge update-available"/>')
                            .text('Update available (installed v' + installedVersion + ')')
                    );
                } else {
                    var badgeText = isActive ? 'Active' : 'Installed';
                    var badgeClass = isActive ? 'active' : 'installed';
                    if (installedVersion) {
                        badgeText += ' (v' + installedVersion + ')';
                    }
                    $status.append(
                        $('<span class="adams-plugin-hub-badge"/>')
                            .addClass(badgeClass)
                            .text(badgeText)
                    );
                }
            } else {
                $status.append(
                    $('<span class="adams-plugin-hub-badge pending"/>')
                        .text('Not installed')
                );
            }

            var $actions = $('<div class="adams-plugin-hub-actions"/>');

            if (!isInstalled) {
                // Fresh install
                var $installBtn = $('<button type="button" class="button button-primary"/>')
                    .text('Install & Activate')
                    .on('click', function() {
                        handleInstall(plugin, $(this), false);
                    });
                $actions.append($installBtn);
            } else {
                // Show an Update button if out of date
                if (hasUpdate) {
                    var label = 'Update to v' + latestVersion;
                    var $updateBtn = $('<button type="button" class="button button-primary"/>')
                        .text(label)
                        .on('click', function() {
                            handleInstall(plugin, $(this), true);
                        });
                    $actions.append($updateBtn);
                }

                // Normal activate/deactivate controls
                if (isActive) {
                    var $deactivateBtn = $('<button type="button" class="button"/>')
                        .text('Deactivate')
                        .on('click', function() {
                            handleToggle(slug, 'deactivate', $(this));
                        });
                    $actions.append($deactivateBtn);
                } else {
                    var $activateBtn = $('<button type="button" class="button"/>')
                        .text('Activate')
                        .on('click', function() {
                            handleToggle(slug, 'activate', $(this));
                        });
                    $actions.append($activateBtn);
                }
            }

            $card.append($header, $desc, $status, $actions);
            $grid.append($card);
        });

        $root.append($grid);
    }

    function handleInstall(plugin, $button, isUpdate) {
        if (!cfg.restUrl) return;

        var originalText = $button.text();
        $button.prop('disabled', true).text(isUpdate ? 'Updating…' : 'Installing…');

        $.ajax({
            url: cfg.restUrl + '/install',
            method: 'POST',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', cfg.nonce);
            },
            data: {
                slug: plugin.slug,
                download_url: plugin.download_url
            }
        }).done(function(res) {
            if (!res || !res.success) {
                alert((isUpdate ? 'Update' : 'Install') + ' failed: ' + (res && res.message ? res.message : 'Unknown error'));
                return;
            }
            // Simpler: reload to refresh installed list / badges
            location.reload();
        }).fail(function() {
            alert((isUpdate ? 'Update' : 'Install') + ' failed. Please try again.');
        }).always(function() {
            $button.prop('disabled', false).text(originalText);
        });
    }

    function handleToggle(slug, action, $button) {
        if (!cfg.restUrl) return;

        var originalText = $button.text();
        $button.prop('disabled', true).text(action === 'activate' ? 'Activating…' : 'Deactivating…');

        $.ajax({
            url: cfg.restUrl + '/toggle',
            method: 'POST',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', cfg.nonce);
            },
            data: {
                slug: slug,
                action: action
            }
        }).done(function(res) {
            if (!res || !res.success) {
                alert('Action failed: ' + (res && res.message ? res.message : 'Unknown error'));
                return;
            }
            location.reload();
        }).fail(function() {
            alert('Action failed. Please try again.');
        }).always(function() {
            $button.prop('disabled', false).text(originalText);
        });
    }

    function loadCatalog() {
        if (!cfg.restUrl) {
            $('#adams-plugin-hub-root').html(
                '<p>REST config missing. Check your Plugin Hub setup.</p>'
            );
            return;
        }

        $('#adams-plugin-hub-root').html(
            '<p>Loading plugin catalog…</p>'
        );

        $.ajax({
            url: cfg.restUrl + '/catalog',
            method: 'GET',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', cfg.nonce);
            }
        }).done(function(res) {
            if (!res || !res.success) {
                $('#adams-plugin-hub-root').html(
                    '<p>Could not load catalog: ' + (res && res.message ? res.message : 'Unknown error') + '</p>'
                );
                return;
            }
            renderPlugins(res.plugins || []);
        }).fail(function() {
            $('#adams-plugin-hub-root').html(
                '<p>Could not connect to plugin catalog. Check your hub API URL.</p>'
            );
        });
    }

    loadCatalog();
});
