<?php
/**
 * Plugin Name: Adam's Safe Updates
 * Description: Safely update all plugins and themes with a pre-update backup, automatic rollback on failure, scheduling, and admin alerts.
 * Version:     0.4.0
 * Author:      Adam Crouchley
 * Author URI:  https://dorimedia.co.nz
 * License:     GPLv2 or later
 * Text Domain: adams-safe-updates
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Adams_Safe_Plugin_Updates {

    const OPTION_LAST_RUN   = 'adams_safe_updates_last_run';
    const OPTION_SETTINGS   = 'adams_safe_updates_settings';
    const BACKUP_DIR        = 'adams-safe-updates';
    const CRON_HOOK         = 'adams_safe_updates_cron';

    /**
     * Bootstrap.
     */
    public static function init() {
        add_action( 'admin_menu', [ __CLASS__, 'add_menu_page' ] );
        add_action( 'admin_init', [ __CLASS__, 'register_settings' ] );
        add_action( 'admin_post_adams_safe_update_plugins', [ __CLASS__, 'handle_safe_update_request' ] );
        add_action( 'admin_notices', [ __CLASS__, 'admin_notices' ] );

        // Admin bar button when there are updates.
        add_action( 'admin_bar_menu', [ __CLASS__, 'admin_bar_node' ], 100 );

        // Extra notice + button on native Updates page.
        add_action( 'load-update-core.php', [ __CLASS__, 'hook_update_core_page' ] );

        // Cron: ensure scheduled event is set/cleared based on settings.
        add_action( 'init', [ __CLASS__, 'ensure_cron_scheduled' ] );
        add_action( self::CRON_HOOK, [ __CLASS__, 'run_scheduled_safe_update' ] );
    }

    /**
     * Add Tools -> Safe Updates page.
     */
    public static function add_menu_page() {
        add_management_page(
            "Adam's Safe Updates",
            'Safe Updates',
            'manage_options',
            'adams-safe-updates',
            [ __CLASS__, 'render_page' ]
        );
    }

    /**
     * Register settings for scheduling.
     */
    public static function register_settings() {
        register_setting(
            'adams_safe_updates',
            self::OPTION_SETTINGS,
            [
                'type'              => 'array',
                'sanitize_callback' => [ __CLASS__, 'sanitize_settings' ],
                'default'           => [],
            ]
        );

        add_settings_section(
            'adams_safe_updates_schedule',
            'Scheduling',
            function () {
                echo '<p>Automatically run Safe Updates on a daily schedule.</p>';
            },
            'adams-safe-updates'
        );

        add_settings_field(
            'adams_safe_updates_schedule_enabled',
            'Enable Scheduled Safe Updates',
            [ __CLASS__, 'render_schedule_field' ],
            'adams-safe-updates',
            'adams_safe_updates_schedule'
        );
    }

    /**
     * Sanitize settings.
     *
     * @param array $input
     * @return array
     */
    public static function sanitize_settings( $input ) {
        $output = [
            'schedule_enabled' => ! empty( $input['schedule_enabled'] ) ? 1 : 0,
        ];

        return $output;
    }

    /**
     * Render the schedule checkbox.
     */
    public static function render_schedule_field() {
        $settings = get_option( self::OPTION_SETTINGS, [] );
        $enabled  = ! empty( $settings['schedule_enabled'] );
        ?>
        <label>
            <input type="checkbox" name="<?php echo esc_attr( self::OPTION_SETTINGS ); ?>[schedule_enabled]" value="1" <?php checked( $enabled ); ?>>
            Run Safe Updates once per day automatically (off-peak, based on site time).
        </label>
        <?php
    }

    /**
     * Render the admin page.
     */
    public static function render_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'adams-safe-updates' ) );
        }

        $last_run = get_option( self::OPTION_LAST_RUN );
        ?>
        <div class="wrap">
            <h1>Adam's Safe Updates</h1>
            <p>This tool will:</p>
            <ol>
                <li>Create a ZIP backup of <code>wp-content/plugins</code> and <code>wp-content/themes</code> (excluding this plugin itself).</li>
                <li>Update plugins and themes <strong>one at a time</strong>, running a health check after each update.</li>
                <li>Automatically roll back all plugin/theme files if the site breaks and record which update most likely caused the issue.</li>
            </ol>

            <?php if ( $last_run && is_array( $last_run ) ) : ?>
                <h2>Last Run</h2>
                <ul>
                    <li><strong>Time:</strong> <?php echo esc_html( $last_run['time'] ); ?></li>
                    <li><strong>Status:</strong> <?php echo esc_html( $last_run['status'] ); ?></li>
                    <?php if ( ! empty( $last_run['message'] ) ) : ?>
                        <li><strong>Message:</strong> <?php echo esc_html( $last_run['message'] ); ?></li>
                    <?php endif; ?>
                    <?php if ( ! empty( $last_run['backup_file'] ) ) : ?>
                        <li><strong>Backup File:</strong> <?php echo esc_html( $last_run['backup_file'] ); ?></li>
                    <?php endif; ?>
                </ul>
            <?php endif; ?>

            <hr>

            <h2>Scheduling</h2>
            <form method="post" action="options.php">
                <?php
                settings_fields( 'adams_safe_updates' );
                do_settings_sections( 'adams-safe-updates' );
                submit_button( 'Save Schedule' );
                ?>
            </form>

            <hr>

            <h2>Run Now</h2>
            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'adams_safe_update_plugins', '_adams_safe_update_nonce' ); ?>
                <input type="hidden" name="action" value="adams_safe_update_plugins">
                <p>
                    <button type="submit" class="button button-primary button-large">
                        Run Safe Update Now (Plugins &amp; Themes)
                    </button>
                </p>
                <p><em>Note: depending on your site and hosting, this may take a little while.</em></p>
            </form>
        </div>
        <?php
    }

    /**
     * Handle the "Run Safe Update" POST request from the admin.
     */
    public static function handle_safe_update_request() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to perform this action.', 'adams-safe-updates' ) );
        }

        check_admin_referer( 'adams_safe_update_plugins', '_adams_safe_update_nonce' );

        $result = self::perform_safe_update();
        self::store_last_run( $result['status'], $result['message'], $result['backup_file'] );

        if ( $result['status'] === 'success' || $result['status'] === 'no_updates' ) {
            self::redirect_with_notice( 'success', $result['message'] );
        } else {
            self::redirect_with_notice( 'error', $result['message'] );
        }

        exit;
    }

    /**
     * Run Safe Updates from cron (scheduled).
     */
    public static function run_scheduled_safe_update() {
        // Double-check scheduling is still enabled.
        $settings = get_option( self::OPTION_SETTINGS, [] );
        if ( empty( $settings['schedule_enabled'] ) ) {
            return;
        }

        $result = self::perform_safe_update();
        self::store_last_run( $result['status'], $result['message'], $result['backup_file'] );

        // Email admin if there are issues (anything other than full success or no updates).
        if ( $result['status'] !== 'success' && $result['status'] !== 'no_updates' ) {
            self::send_issue_email( $result );
        }
    }

    /**
     * Core Safe Update routine shared by manual + cron.
     * Updates plugins and themes one-by-one, with a health check after each.
     *
     * @return array {status, message, backup_file}
     */
    protected static function perform_safe_update() {
        // 0) Gather update lists first. If nothing to do, bail early without backup.
        $plugins_to_update = self::get_plugin_update_list();
        $themes_to_update  = self::get_theme_update_list();

        if ( empty( $plugins_to_update ) && empty( $themes_to_update ) ) {
            return [
                'status'      => 'no_updates',
                'message'     => 'No plugins or themes to update.',
                'backup_file' => '',
            ];
        }

        $result_status  = 'unknown';
        $result_message = '';
        $backup_file    = '';

        // 1) Create backup of plugins + themes.
        $backup_file = self::create_code_backup();
        if ( ! $backup_file ) {
            $result_status  = 'failed';
            $result_message = 'Failed to create plugins/themes backup. Updates were NOT run.';
            return [
                'status'      => $result_status,
                'message'     => $result_message,
                'backup_file' => '',
            ];
        }

        // 2) Update plugins one-by-one.
        foreach ( $plugins_to_update as $plugin_file ) {
            $upgrade_result = self::upgrade_single_plugin( $plugin_file );
            if ( is_wp_error( $upgrade_result ) ) {
                $result_status  = 'failed';
                $result_message = 'Plugin update failed for ' . $plugin_file . ': ' . $upgrade_result->get_error_message();
                // No rollback yet: site may still be healthy but plugin just refused to update.
                // However, safest is to roll back everything to pre-run state.
                self::restore_code_backup( $backup_file );
                return [
                    'status'      => $result_status,
                    'message'     => $result_message . ' All plugin/theme files have been restored from backup.',
                    'backup_file' => $backup_file,
                ];
            }

            // Health check after each plugin update.
            $health_ok = self::run_health_check( $health_reason );
            if ( ! $health_ok ) {
                self::restore_code_backup( $backup_file );
                $result_status  = 'rolled_back';
                $result_message = 'Health check failed after updating plugin ' . $plugin_file . ' (' . $health_reason . '). Backup restored. Likely culprit: ' . $plugin_file;
                return [
                    'status'      => $result_status,
                    'message'     => $result_message,
                    'backup_file' => $backup_file,
                ];
            }
        }

        // 3) Update themes one-by-one.
        foreach ( $themes_to_update as $theme_stylesheet ) {
            $upgrade_result = self::upgrade_single_theme( $theme_stylesheet );
            if ( is_wp_error( $upgrade_result ) ) {
                $result_status  = 'failed';
                $result_message = 'Theme update failed for ' . $theme_stylesheet . ': ' . $upgrade_result->get_error_message();
                self::restore_code_backup( $backup_file );
                return [
                    'status'      => $result_status,
                    'message'     => $result_message . ' All plugin/theme files have been restored from backup.',
                    'backup_file' => $backup_file,
                ];
            }

            // Health check after each theme update.
            $health_ok = self::run_health_check( $health_reason );
            if ( ! $health_ok ) {
                self::restore_code_backup( $backup_file );
                $result_status  = 'rolled_back';
                $result_message = 'Health check failed after updating theme ' . $theme_stylesheet . ' (' . $health_reason . '). Backup restored. Likely culprit: ' . $theme_stylesheet;
                return [
                    'status'      => $result_status,
                    'message'     => $result_message,
                    'backup_file' => $backup_file,
                ];
            }
        }

        // 4) If we got here, all updates passed individual health checks.
        $result_status  = 'success';
        $result_message = 'Plugins and themes updated successfully. All per-update health checks passed.';
        return [
            'status'      => $result_status,
            'message'     => $result_message,
            'backup_file' => $backup_file,
        ];
    }

    /**
     * Get list of plugin files that have an update available.
     *
     * @return array
     */
    protected static function get_plugin_update_list() {
        if ( ! function_exists( 'wp_update_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/update.php';
        }

        wp_update_plugins();
        $updates = get_site_transient( 'update_plugins' );

        if ( empty( $updates->response ) || ! is_array( $updates->response ) ) {
            return [];
        }

        return array_keys( $updates->response );
    }

    /**
     * Get list of theme stylesheets that have an update available.
     *
     * @return array
     */
    protected static function get_theme_update_list() {
        if ( ! function_exists( 'wp_update_themes' ) ) {
            require_once ABSPATH . 'wp-admin/includes/update.php';
        }

        wp_update_themes();
        $updates = get_site_transient( 'update_themes' );

        if ( empty( $updates->response ) || ! is_array( $updates->response ) ) {
            return [];
        }

        return array_keys( $updates->response );
    }

    /**
     * Upgrade a single plugin.
     *
     * @param string $plugin_file
     * @return true|\WP_Error
     */
    protected static function upgrade_single_plugin( $plugin_file ) {
        if ( ! class_exists( 'Plugin_Upgrader' ) ) {
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        }

        $skin     = new Automatic_Upgrader_Skin();
        $upgrader = new Plugin_Upgrader( $skin );

        $result = $upgrader->upgrade( $plugin_file );

        if ( $result === false ) {
            return new WP_Error( 'plugin_upgrade_failed', 'Unknown error upgrading plugin ' . $plugin_file );
        }

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return true;
    }

    /**
     * Upgrade a single theme.
     *
     * @param string $theme_stylesheet
     * @return true|\WP_Error
     */
    protected static function upgrade_single_theme( $theme_stylesheet ) {
        if ( ! class_exists( 'Theme_Upgrader' ) ) {
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        }

        $skin     = new Automatic_Upgrader_Skin();
        $upgrader = new Theme_Upgrader( $skin );

        $result = $upgrader->upgrade( $theme_stylesheet );

        if ( $result === false ) {
            return new WP_Error( 'theme_upgrade_failed', 'Unknown error upgrading theme ' . $theme_stylesheet );
        }

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return true;
    }

    /**
     * Create a ZIP backup of wp-content/plugins and wp-content/themes
     * (excluding this plugin's own folder to avoid overwriting itself).
     *
     * @return string|false Backup file path or false on failure.
     */
    protected static function create_code_backup() {
        if ( ! class_exists( 'ZipArchive' ) ) {
            return false; // ZipArchive extension required.
        }

        $plugins_dir = WP_PLUGIN_DIR;
        $themes_root = get_theme_root();

        $uploads    = wp_upload_dir();
        $backup_dir = trailingslashit( $uploads['basedir'] ) . self::BACKUP_DIR;

        if ( ! wp_mkdir_p( $backup_dir ) ) {
            return false;
        }

        $timestamp   = date( 'Ymd-His' );
        $backup_file = trailingslashit( $backup_dir ) . "code-backup-{$timestamp}.zip";

        $zip = new ZipArchive();
        if ( true !== $zip->open( $backup_file, ZipArchive::CREATE | ZipArchive::OVERWRITE ) ) {
            return false;
        }

        $self_dir_name = basename( __DIR__ );

        // Backup plugins.
        if ( is_dir( $plugins_dir ) ) {
            $dir_iterator = new RecursiveDirectoryIterator( $plugins_dir, FilesystemIterator::SKIP_DOTS );
            $iterator     = new RecursiveIteratorIterator( $dir_iterator, RecursiveIteratorIterator::LEAVES_ONLY );

            foreach ( $iterator as $file ) {
                /** @var SplFileInfo $file */
                if ( $file->isDir() ) {
                    continue;
                }

                $file_path = $file->getRealPath();
                $relative  = substr( $file_path, strlen( $plugins_dir ) + 1 );

                // Skip this plugin's folder to avoid overwriting the tool itself.
                if ( strpos( $relative, $self_dir_name . DIRECTORY_SEPARATOR ) === 0 ) {
                    continue;
                }

                $zip->addFile( $file_path, 'plugins/' . $relative );
            }
        }

        // Backup themes.
        if ( is_dir( $themes_root ) ) {
            $themes_iterator = new RecursiveDirectoryIterator( $themes_root, FilesystemIterator::SKIP_DOTS );
            $iterator2       = new RecursiveIteratorIterator( $themes_iterator, RecursiveIteratorIterator::LEAVES_ONLY );

            foreach ( $iterator2 as $file ) {
                /** @var SplFileInfo $file */
                if ( $file->isDir() ) {
                    continue;
                }

                $file_path = $file->getRealPath();
                $relative  = substr( $file_path, strlen( $themes_root ) + 1 );

                $zip->addFile( $file_path, 'themes/' . $relative );
            }
        }

        $zip->close();

        return $backup_file;
    }

    /**
     * Restore plugins and themes from a ZIP backup created by create_code_backup().
     *
     * @param string $backup_file
     * @return bool
     */
    protected static function restore_code_backup( $backup_file ) {
        if ( ! class_exists( 'ZipArchive' ) || ! file_exists( $backup_file ) ) {
            return false;
        }

        $zip = new ZipArchive();
        if ( true !== $zip->open( $backup_file ) ) {
            return false;
        }

        // Extract only into the correct roots.
        // Our backup stored files under "plugins/" and "themes/" folders inside the ZIP.
        for ( $i = 0; $i < $zip->numFiles; $i++ ) {
            $entry = $zip->getNameIndex( $i );
            if ( strpos( $entry, 'plugins/' ) === 0 ) {
                $relative = substr( $entry, strlen( 'plugins/' ) );
                $target   = trailingslashit( WP_PLUGIN_DIR ) . $relative;

                // Ensure directory exists.
                $dir = dirname( $target );
                if ( ! is_dir( $dir ) ) {
                    wp_mkdir_p( $dir );
                }

                copy( "zip://{$backup_file}#" . $entry, $target );
            } elseif ( strpos( $entry, 'themes/' ) === 0 ) {
                $relative = substr( $entry, strlen( 'themes/' ) );
                $target   = trailingslashit( get_theme_root() ) . $relative;

                $dir = dirname( $target );
                if ( ! is_dir( $dir ) ) {
                    wp_mkdir_p( $dir );
                }

                copy( "zip://{$backup_file}#" . $entry, $target );
            }
        }

        $zip->close();

        return true;
    }

    /**
     * Run a basic health check:
     * - GET home_url('/')
     * - Check for HTTP 2xx and absence of obvious "Fatal error" text.
     *
     * @param string $reason Reason text (output parameter).
     * @return bool True if healthy, false if not.
     */
    protected static function run_health_check( &$reason = '' ) {
        $url      = home_url( '/' );
        $response = wp_remote_get( $url, [
            'timeout'   => 20,
            'sslverify' => false,
        ] );

        if ( is_wp_error( $response ) ) {
            $reason = 'HTTP request error: ' . $response->get_error_message();
            return false;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( $code < 200 || $code >= 300 ) {
            $reason = 'Non-2xx response from home page: ' . $code;
            return false;
        }

        $body = wp_remote_retrieve_body( $response );
        if (
            stripos( $body, 'fatal error' ) !== false
            || stripos( $body, 'there has been a critical error on this website' ) !== false
        ) {
            $reason = 'Fatal error text detected on home page.';
            return false;
        }

        $reason = 'OK';
        return true;
    }

    /**
     * Store last run details.
     */
    protected static function store_last_run( $status, $message, $backup_file ) {
        $data = [
            'time'        => current_time( 'mysql' ),
            'status'      => $status,
            'message'     => $message,
            'backup_file' => $backup_file,
        ];
        update_option( self::OPTION_LAST_RUN, $data, false );
    }

    /**
     * Redirect back to settings page with a notice flag.
     */
    protected static function redirect_with_notice( $type, $message ) {
        $query_args = [
            'page'                  => 'adams-safe-updates',
            'adams_safe_notice'     => $type,
            'adams_safe_notice_msg' => rawurlencode( $message ),
        ];
        $url = add_query_arg( $query_args, admin_url( 'tools.php' ) );
        wp_safe_redirect( $url );
        exit;
    }

    /**
     * Show admin notice after redirect (success/error message on our page).
     */
    public static function admin_notices() {
        if ( isset( $_GET['adams_safe_notice'], $_GET['adams_safe_notice_msg'] ) ) {
            $type    = sanitize_key( wp_unslash( $_GET['adams_safe_notice'] ) );
            $message = rawurldecode( wp_unslash( $_GET['adams_safe_notice_msg'] ) );

            $class = ( $type === 'success' ) ? 'notice-success' : 'notice-error';

            echo '<div class="notice ' . esc_attr( $class ) . ' is-dismissible">';
            echo '<p>' . esc_html( $message ) . '</p>';
            echo '</div>';
        }
    }

    /**
     * Determine if there are any plugin/theme updates available.
     *
     * @return bool
     */
    protected static function has_updates() {
        if ( ! function_exists( 'get_site_transient' ) ) {
            return false;
        }

        $plugin_updates = get_site_transient( 'update_plugins' );
        $theme_updates  = get_site_transient( 'update_themes' );

        $has_plugin = ( isset( $plugin_updates->response ) && is_array( $plugin_updates->response ) && count( $plugin_updates->response ) > 0 );
        $has_theme  = ( isset( $theme_updates->response ) && is_array( $theme_updates->response ) && count( $theme_updates->response ) > 0 );

        return ( $has_plugin || $has_theme );
    }

    /**
     * Add a toolbar button when there are updates.
     *
     * @param WP_Admin_Bar $wp_admin_bar
     */
    public static function admin_bar_node( $wp_admin_bar ) {
        if ( ! is_admin_bar_showing() ) {
            return;
        }

        if ( ! current_user_can( 'update_plugins' ) && ! current_user_can( 'update_themes' ) ) {
            return;
        }

        if ( ! self::has_updates() ) {
            return;
        }

        $wp_admin_bar->add_node( [
            'id'    => 'adams-safe-updates',
            'title' => 'Safe Update',
            'href'  => admin_url( 'tools.php?page=adams-safe-updates' ),
            'meta'  => [
                'class' => 'adams-safe-updates-toolbar',
                'title' => "Run Adam's Safe Updates (plugins & themes)",
            ],
        ] );
    }

    /**
     * Hooked on load-update-core.php to add a notice on the native Updates page.
     */
    public static function hook_update_core_page() {
        add_action( 'admin_notices', [ __CLASS__, 'render_update_core_notice' ] );
    }

    /**
     * Render a notice with button on the core Updates page.
     */
    public static function render_update_core_notice() {
        if ( ! current_user_can( 'update_plugins' ) && ! current_user_can( 'update_themes' ) ) {
            return;
        }

        if ( ! self::has_updates() ) {
            return;
        }

        $url = admin_url( 'tools.php?page=adams-safe-updates' );
        ?>
        <div class="notice notice-info adams-safe-updates-notice">
            <p><strong>Adam's Safe Updates:</strong> You have plugin/theme updates available. You can run them with a backup and automatic rollback.</p>
            <p>
                <a href="<?php echo esc_url( $url ); ?>" class="button button-primary">
                    Run Safe Update (Plugins &amp; Themes)
                </a>
            </p>
        </div>
        <?php
    }

    /**
     * Ensure the cron event is scheduled/unscheduled based on settings.
     */
    public static function ensure_cron_scheduled() {
        if ( ! function_exists( 'wp_next_scheduled' ) ) {
            return;
        }

        $settings = get_option( self::OPTION_SETTINGS, [] );
        $enabled  = ! empty( $settings['schedule_enabled'] );
        $next     = wp_next_scheduled( self::CRON_HOOK );

        if ( $enabled && ! $next ) {
            // Schedule daily, starting in one hour from now.
            wp_schedule_event( time() + HOUR_IN_SECONDS, 'daily', self::CRON_HOOK );
        } elseif ( ! $enabled && $next ) {
            wp_unschedule_event( $next, self::CRON_HOOK );
        }
    }

    /**
     * Send an email to the admin if a scheduled run has issues.
     *
     * @param array $result
     */
    protected static function send_issue_email( $result ) {
        if ( ! function_exists( 'wp_mail' ) ) {
            return;
        }

        $admin_email = get_option( 'admin_email' );
        if ( ! $admin_email ) {
            return;
        }

        $site_name = get_bloginfo( 'name' );
        $subject   = sprintf( "[Adam's Safe Updates] Issue on %s", $site_name );

        $lines = [];
        $lines[] = 'A scheduled Safe Updates run has completed with issues.';
        $lines[] = '';
        $lines[] = 'Status: ' . $result['status'];
        $lines[] = 'Message: ' . $result['message'];
        if ( ! empty( $result['backup_file'] ) ) {
            $lines[] = 'Backup file: ' . $result['backup_file'];
        }
        $lines[] = '';
        $lines[] = 'Site: ' . home_url( '/' );
        $lines[] = '';
        $lines[] = 'You may need to review the site and/or restore from backup manually if problems persist.';

        $body = implode( "\n", $lines );

        wp_mail( $admin_email, $subject, $body );
    }
}

Adams_Safe_Plugin_Updates::init();
