<?php
/**
 * Plugin Name: Adam's Security Manager
 * Description: Lightweight security hardening for WordPress – headers, CSP, XML-RPC, login protection, and general hardening.
 * Author: Dori Media / Adam Crouchley
 * Version: 1.2.2
 * License: GPL2+
 * Text Domain: adams-security-manager
 */

if (!defined('ABSPATH')) {
    exit;
}

class SM_Security_Manager {

    const OPTION_KEY = 'sm_security_manager_options';
    const CRON_HOOK  = 'sm_security_manager_daily_scan';

    private static $instance = null;

    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // Core security tweaks
        add_action('init', array($this, 'apply_wp_constants'), 1);
        add_action('init', array($this, 'disable_xmlrpc'));
        add_action('init', array($this, 'hide_wp_version'));
        add_action('init', array($this, 'disable_feeds'));
        add_action('init', array($this, 'disable_emojis'));
        add_action('init', array($this, 'disable_oembed'));
        add_action('init', array($this, 'disable_sitemaps'));
        add_action('rest_api_init', array($this, 'disable_rest_users'), 11);
        add_action('template_redirect', array($this, 'block_author_enum'));

        // Login hardening
        add_filter('login_errors', array($this, 'hide_login_errors'));
        add_action('login_init', array($this, 'maybe_block_default_login'));
        add_filter('login_url', array($this, 'filter_login_url'), 10, 3);
        add_action('init', array($this, 'add_custom_login_rewrite'));
        add_action('template_redirect', array($this, 'render_custom_login_template'));
        add_action('login_form', array($this, 'render_turnstile_on_login'));
        add_filter('authenticate', array($this, 'validate_turnstile_on_login'), 20, 3);
        add_filter('authenticate', array($this, 'rate_limit_authenticate'), 30, 3);

        // Strong passwords
        add_action('user_profile_update_errors', array($this, 'enforce_strong_passwords'), 10, 3);

        // Security headers
        add_action('send_headers', array($this, 'send_security_headers'));

        // Admin UI
        if (is_admin()) {
            add_action('admin_menu', array($this, 'register_settings_page'));
            add_action('admin_init', array($this, 'register_settings'));
        }

        // Cron: daily security/update scan
        add_action(self::CRON_HOOK, array($this, 'run_daily_scan'));
    }

    /* ========================================================
     * Activation / Deactivation
     * ===================================================== */

    public function activate() {
        $defaults = array(
            'enable_headers'       => 1,
            'enable_csp'           => 1,
            'csp_policy'           => $this->get_default_csp(),
            'disable_xmlrpc'       => 1,
            'block_author_enum'    => 1,
            'hide_wp_version'      => 1,
            'disable_feeds'        => 1,
            'disable_emojis'       => 1,
            'disable_oembed'       => 1,
            'disable_rest_users'   => 1,
            'disable_sitemaps'     => 0, // some sites use core sitemaps
            'disable_file_edit'    => 1,
            'enforce_strong_pw'    => 1,
            'hide_login_errors'    => 1,

            // Login URL + rate limiting
            'move_login_url_enabled' => 0,
            'login_slug'             => 'secure-login',
            'rate_limit_enabled'     => 1,
            'rate_limit_max_attempts'=> 5,
            'rate_limit_window'      => 900,   // 15 minutes
            'rate_limit_lockout'     => 900,   // 15 minutes

            // Cloudflare Turnstile
            'turnstile_enabled'      => 0,
            'turnstile_site_key'     => '',
            'turnstile_secret_key'   => '',

            // Daily scan
            'daily_scan_enabled'     => 1,
            'daily_scan_email'       => get_option('admin_email'),
        );

        $current = get_option(self::OPTION_KEY, array());
        update_option(self::OPTION_KEY, wp_parse_args($current, $defaults));

        // Setup cron for daily scan
        if (!wp_next_scheduled(self::CRON_HOOK)) {
            wp_schedule_event(time() + 3600, 'daily', self::CRON_HOOK);
        }

        // Add custom login rewrite
        $this->add_custom_login_rewrite();
        flush_rewrite_rules();
    }

    public function deactivate() {
        // Remove cron
        $timestamp = wp_next_scheduled(self::CRON_HOOK);
        if ($timestamp) {
            wp_unschedule_event($timestamp, self::CRON_HOOK);
        }
        flush_rewrite_rules();
    }

    private function get_options() {
        $opts = get_option(self::OPTION_KEY, array());
        $defaults = array(
            'enable_headers'       => 1,
            'enable_csp'           => 1,
            'csp_policy'           => $this->get_default_csp(),
            'disable_xmlrpc'       => 1,
            'block_author_enum'    => 1,
            'hide_wp_version'      => 1,
            'disable_feeds'        => 1,
            'disable_emojis'       => 1,
            'disable_oembed'       => 1,
            'disable_rest_users'   => 1,
            'disable_sitemaps'     => 0,
            'disable_file_edit'    => 1,
            'enforce_strong_pw'    => 1,
            'hide_login_errors'    => 1,

            'move_login_url_enabled' => 0,
            'login_slug'             => 'secure-login',
            'rate_limit_enabled'     => 1,
            'rate_limit_max_attempts'=> 5,
            'rate_limit_window'      => 900,
            'rate_limit_lockout'     => 900,

            'turnstile_enabled'      => 0,
            'turnstile_site_key'     => '',
            'turnstile_secret_key'   => '',

            'daily_scan_enabled'     => 1,
            'daily_scan_email'       => get_option('admin_email'),
        );
        return wp_parse_args($opts, $defaults);
    }

    /**
     * Default CSP:
     * - Works with Gutenberg & Elementor (blob:, data:, unsafe-eval)
     * - Allows Google Fonts (fonts.googleapis.com / fonts.gstatic.com)
     * - Allows GA4 / GTM (googletagmanager / google-analytics / g.doubleclick)
     * - Allows Cloudflare Insights (static.cloudflareinsights.com)
     */
    private function get_default_csp() {
        return
            "default-src 'self' https: data: blob:; " .
            "script-src 'self' https: 'unsafe-inline' 'unsafe-eval' blob: https://www.googletagmanager.com https://www.google-analytics.com https://stats.g.doubleclick.net https://static.cloudflareinsights.com; " .
            "style-src 'self' 'unsafe-inline' https: blob: https://fonts.googleapis.com; " .
            "img-src 'self' data: blob: https:; " .
            "font-src 'self' data: https: https://fonts.gstatic.com; " .
            "connect-src 'self' https: wss: https://www.google-analytics.com https://region1.google-analytics.com https://stats.g.doubleclick.net https://static.cloudflareinsights.com; " .
            "frame-src 'self' https: blob: https://www.youtube.com https://player.vimeo.com; " .
            "worker-src 'self' blob:; " .
            "media-src 'self' data: blob: https:; " .
            "object-src 'none';";
    }

    /* ========================================================
     * Core WP-level tweaks
     * ===================================================== */

    // Apply WP constants (file editing etc.)
    public function apply_wp_constants() {
        $opts = $this->get_options();

        if (!empty($opts['disable_file_edit'])) {
            if (!defined('DISALLOW_FILE_EDIT')) {
                define('DISALLOW_FILE_EDIT', true);
            }
            if (!defined('DISALLOW_FILE_MODS')) {
                define('DISALLOW_FILE_MODS', true);
            }
        }
    }

    public function disable_xmlrpc() {
        $opts = $this->get_options();
        if (!empty($opts['disable_xmlrpc'])) {
            add_filter('xmlrpc_enabled', '__return_false');
        }
    }

    public function hide_wp_version() {
        $opts = $this->get_options();
        if (!empty($opts['hide_wp_version'])) {
            remove_action('wp_head', 'wp_generator');
        }
    }

    public function disable_feeds() {
        $opts = $this->get_options();
        if (empty($opts['disable_feeds'])) {
            return;
        }

        foreach (array(
            'do_feed',
            'do_feed_rdf',
            'do_feed_rss',
            'do_feed_rss2',
            'do_feed_atom',
            'do_feed_rss2_comments',
            'do_feed_atom_comments',
        ) as $feed_hook) {
            add_action($feed_hook, function () {
                wp_die(__('No feed available.', 'sm-security'), '', array('response' => 404));
            }, 1);
        }

        // Remove feed links from head
        remove_action('wp_head', 'feed_links_extra', 3);
        remove_action('wp_head', 'feed_links', 2);
    }

    public function disable_emojis() {
        $opts = $this->get_options();
        if (empty($opts['disable_emojis'])) {
            return;
        }

        remove_action('wp_head', 'print_emoji_detection_script', 7);
        remove_action('admin_print_scripts', 'print_emoji_detection_script');
        remove_action('wp_print_styles', 'print_emoji_styles');
        remove_action('admin_print_styles', 'print_emoji_styles');
        remove_filter('the_content_feed', 'wp_staticize_emoji');
        remove_filter('comment_text_rss', 'wp_staticize_emoji');
        remove_filter('wp_mail', 'wp_staticize_emoji_for_email');
    }

    public function disable_oembed() {
        $opts = $this->get_options();
        if (empty($opts['disable_oembed'])) {
            return;
        }

        remove_action('rest_api_init', 'wp_oembed_register_route');
        remove_filter('oembed_dataparse', 'wp_filter_oembed_result', 10);
        remove_action('wp_head', 'wp_oembed_add_discovery_links');
        remove_action('wp_head', 'wp_oembed_add_host_js');
    }

    public function disable_sitemaps() {
        $opts = $this->get_options();
        if (empty($opts['disable_sitemaps'])) {
            return;
        }
        add_filter('wp_sitemaps_enabled', '__return_false');
    }

    /**
     * Disable REST user endpoints for logged-out visitors only.
     * Keeps REST intact for admin/editor so Gutenberg/Elementor keep working.
     */
    public function disable_rest_users() {
        $opts = $this->get_options();
        if (empty($opts['disable_rest_users'])) {
            return;
        }

        add_filter('rest_endpoints', function ($endpoints) {
            // Only affect unauthenticated requests.
            // When logged in (e.g. in editor/admin), leave endpoints alone.
            if (is_user_logged_in()) {
                return $endpoints;
            }

            if (isset($endpoints['/wp/v2/users'])) {
                unset($endpoints['/wp/v2/users']);
            }
            if (isset($endpoints['/wp/v2/users/(?P<id>[\d]+)'])) {
                unset($endpoints['/wp/v2/users/(?P<id>[\d]+)']);
            }

            return $endpoints;
        });
    }

    // Block ?author=1 enumeration
    public function block_author_enum() {
        $opts = $this->get_options();
        if (empty($opts['block_author_enum'])) {
            return;
        }

        if (is_admin()) {
            return;
        }

        if (isset($_GET['author']) && is_numeric($_GET['author'])) {
            global $wp_query;
            $wp_query->set_404();
            status_header(404);
            nocache_headers();
        }
    }

    /* ========================================================
     * Login hardening
     * ===================================================== */

    // Generic login error to avoid info leaks
    public function hide_login_errors($error) {
        $opts = $this->get_options();
        if (empty($opts['hide_login_errors'])) {
            return $error;
        }
        return __('Invalid credentials.', 'sm-security');
    }

    // Custom login URL rewrite
    public function add_custom_login_rewrite() {
        $opts = $this->get_options();
        if (empty($opts['move_login_url_enabled']) || empty($opts['login_slug'])) {
            return;
        }

        add_rewrite_tag('%sm_custom_login%', '1');

        add_rewrite_rule(
            '^' . preg_quote($opts['login_slug'], '/') . '/?$',
            'index.php?sm_custom_login=1',
            'top'
        );
    }

    // Render login form at custom login URL
    public function render_custom_login_template() {
        if (get_query_var('sm_custom_login')) {
            // Treat as login
            require_once ABSPATH . 'wp-login.php';
            exit;
        }
    }

    // Filter login_url to use custom slug
    public function filter_login_url($login_url, $redirect, $force_reauth) {
        $opts = $this->get_options();
        if (!empty($opts['move_login_url_enabled']) && !empty($opts['login_slug'])) {
            $login_url = home_url('/' . $opts['login_slug'] . '/');
            if (!empty($redirect)) {
                $login_url = add_query_arg('redirect_to', urlencode($redirect), $login_url);
            }
        }
        return $login_url;
    }

    // Block direct access to wp-login.php when custom login is enabled
    public function maybe_block_default_login() {
        $opts = $this->get_options();
        if (empty($opts['move_login_url_enabled'])) {
            return;
        }

        $action = isset($_REQUEST['action']) ? sanitize_text_field($_REQUEST['action']) : 'login';

        // Allow these core actions even if default URL is hit
        $allowed_actions = array('logout', 'lostpassword', 'rp', 'resetpass', 'postpass');

        if (!in_array($action, $allowed_actions, true)) {
            // If URL doesn't contain our custom slug, block direct wp-login.php access
            $request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
            if (strpos($request_uri, '/' . $opts['login_slug'] . '/') === false) {
                wp_die(__('Login is not available at this URL.', 'sm-security'), 403);
            }
        }
    }

    // Rate limiting by IP on authenticate
    public function rate_limit_authenticate($user, $username, $password) {
        $opts = $this->get_options();
        if (empty($opts['rate_limit_enabled'])) {
            return $user;
        }

        $ip = $this->get_client_ip();
        if (!$ip) {
            return $user;
        }

        $max_attempts = (int) $opts['rate_limit_max_attempts'];
        $window       = (int) $opts['rate_limit_window'];
        $lockout      = (int) $opts['rate_limit_lockout'];

        $key = 'sm_login_attempts_' . md5($ip);
        $data = get_transient($key);
        if (!is_array($data)) {
            $data = array(
                'count'    => 0,
                'first'    => time(),
                'locked'   => 0,
            );
        }

        // If locked and lockout not expired
        if (!empty($data['locked']) && time() < $data['locked']) {
            return new WP_Error(
                'sm_rate_limited',
                __('Too many failed login attempts. Please try again later.', 'sm-security')
            );
        }

        // Only count failures; success resets counter (below)
        if ($user instanceof WP_Error || (empty($username) || empty($password))) {
            // New window?
            if ((time() - $data['first']) > $window) {
                $data['count'] = 0;
                $data['first'] = time();
            }

            $data['count']++;

            if ($data['count'] >= $max_attempts) {
                $data['locked'] = time() + $lockout;
            }

            set_transient($key, $data, max($window, $lockout));

            if (!empty($data['locked']) && time() < $data['locked']) {
                return new WP_Error(
                    'sm_rate_limited',
                    __('Too many failed login attempts. Please try again later.', 'sm-security')
                );
            }
        } else {
            // Successful login: reset attempts
            delete_transient($key);
        }

        return $user;
    }

    private function get_client_ip() {
        $keys = array(
            'HTTP_CF_CONNECTING_IP', // Cloudflare
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'REMOTE_ADDR',
        );
        foreach ($keys as $k) {
            if (!empty($_SERVER[$k])) {
                $ip = explode(',', $_SERVER[$k]);
                return trim($ip[0]);
            }
        }
        return null;
    }

    /* ========================================================
     * Strong password enforcement
     * ===================================================== */

    public function enforce_strong_passwords($errors, $update, $user) {
        $opts = $this->get_options();
        if (empty($opts['enforce_strong_pw'])) {
            return;
        }

        if (empty($_POST['pass1'])) {
            return;
        }

        $password = (string) $_POST['pass1'];

        // Only enforce for higher privilege roles
        if (!empty($_POST['role'])) {
            $role = sanitize_text_field(wp_unslash($_POST['role']));
        } elseif (!empty($user->roles)) {
            $role = reset($user->roles);
        } else {
            $role = '';
        }

        $sensitive_roles = array('administrator', 'editor', 'shop_manager');

        if (!in_array($role, $sensitive_roles, true)) {
            return;
        }

        // Simple strength rules: 12+ chars, letters + numbers
        if (strlen($password) < 12 ||
            !preg_match('/[A-Za-z]/', $password) ||
            !preg_match('/\d/', $password)
        ) {
            $errors->add(
                'weak_password',
                __('Please use a stronger password (min 12 characters with letters and numbers).', 'sm-security')
            );
        }
    }

    /* ========================================================
     * Cloudflare Turnstile
     * ===================================================== */

    public function render_turnstile_on_login() {
        $opts = $this->get_options();
        if (empty($opts['turnstile_enabled']) || empty($opts['turnstile_site_key'])) {
            return;
        }
        ?>
        <script src="https://challenges.cloudflare.com/turnstile/v0/api.js" async defer></script>
        <div class="cf-turnstile"
             data-sitekey="<?php echo esc_attr($opts['turnstile_site_key']); ?>"
             style="margin-bottom: 1em;"></div>
        <?php
    }

    public function validate_turnstile_on_login($user, $username, $password) {
        $opts = $this->get_options();
        if (empty($opts['turnstile_enabled']) || empty($opts['turnstile_secret_key'])) {
            return $user;
        }

        // Only validate when both username + password filled
        if (empty($username) || empty($password)) {
            return $user;
        }

        if (!isset($_POST['cf-turnstile-response'])) {
            return new WP_Error('sm_turnstile_missing', __('Please complete the CAPTCHA.', 'sm-security'));
        }

        $response = sanitize_text_field(wp_unslash($_POST['cf-turnstile-response']));

        $verify = wp_remote_post('https://challenges.cloudflare.com/turnstile/v0/siteverify', array(
            'timeout' => 10,
            'body'    => array(
                'secret'   => $opts['turnstile_secret_key'],
                'response' => $response,
                'remoteip' => $this->get_client_ip(),
            ),
        ));

        if (is_wp_error($verify)) {
            return new WP_Error('sm_turnstile_error', __('CAPTCHA verification failed. Please try again.', 'sm-security'));
        }

        $data = json_decode(wp_remote_retrieve_body($verify), true);
        if (empty($data['success'])) {
            return new WP_Error('sm_turnstile_invalid', __('CAPTCHA verification failed. Please try again.', 'sm-security'));
        }

        return $user;
    }

    /* ========================================================
     * Security headers
     * ===================================================== */

    public function send_security_headers() {
        $opts = $this->get_options();

        if (headers_sent() || empty($opts['enable_headers'])) {
            return;
        }

        header('X-Frame-Options: SAMEORIGIN', true);
        header('X-Content-Type-Options: nosniff', true);
        header('Referrer-Policy: strict-origin-when-cross-origin', true);
        header('X-XSS-Protection: 1; mode=block', true);

        if (!empty($opts['enable_csp'])) {
            $csp = trim($opts['csp_policy']);
            if (!empty($csp)) {
                header('Content-Security-Policy: ' . $csp, true);
            }
        }
    }

    /* ========================================================
     * Daily scan (updates)
     * ===================================================== */

    public function run_daily_scan() {
        $opts = $this->get_options();
        if (empty($opts['daily_scan_enabled'])) {
            return;
        }

        $email = isset($opts['daily_scan_email']) ? sanitize_email($opts['daily_scan_email']) : '';
        if (empty($email)) {
            return;
        }

        // Ensure update data is fresh
        if (function_exists('wp_update_plugins')) {
            wp_update_plugins();
        }
        if (function_exists('wp_update_themes')) {
            wp_update_themes();
        }
        if (function_exists('wp_version_check')) {
            wp_version_check();
        }

        $updates = array(
            'core'    => $this->get_core_updates_summary(),
            'plugins' => $this->get_plugin_updates_summary(),
            'themes'  => $this->get_theme_updates_summary(),
        );

        if (empty($updates['core']) && empty($updates['plugins']) && empty($updates['themes'])) {
            return;
        }

        $subject = sprintf('[%s] WordPress updates available', wp_specialchars_decode(get_bloginfo('name'), ENT_QUOTES));
        $lines   = array();

        if (!empty($updates['core'])) {
            $lines[] = 'Core: ' . $updates['core'];
        }
        if (!empty($updates['plugins'])) {
            $lines[] = 'Plugins: ' . $updates['plugins'];
        }
        if (!empty($updates['themes'])) {
            $lines[] = 'Themes: ' . $updates['themes'];
        }

        $body = "Adam's Security Manager detected updates available on " . site_url() . ":\n\n";
        $body .= implode("\n", $lines) . "\n\n";
        $body .= "Please log in and update when convenient.\n";

        wp_mail($email, $subject, $body);
    }

    private function get_core_updates_summary() {
        $updates = get_core_updates();
        if (empty($updates) || !is_array($updates)) {
            return '';
        }
        foreach ($updates as $u) {
            if (!empty($u->response) && $u->response === 'upgrade') {
                return 'Update available to ' . $u->version . '.';
            }
        }
        return '';
    }

    private function get_plugin_updates_summary() {
        $updates = get_site_transient('update_plugins');
        if (empty($updates->response) || !is_array($updates->response)) {
            return '';
        }
        return count($updates->response) . ' plugin(s) have updates.';
    }

    private function get_theme_updates_summary() {
        $updates = get_site_transient('update_themes');
        if (empty($updates->response) || !is_array($updates->response)) {
            return '';
        }
        return count($updates->response) . ' theme(s) have updates.';
    }

    /* ========================================================
     * Settings page
     * ===================================================== */

    public function register_settings_page() {
        add_options_page(
            "Adam's Security Manager",
            "Adam's Security Manager",
            'manage_options',
            'sm-security-manager',
            array($this, 'render_settings_page')
        );
    }

    public function register_settings() {
        register_setting(
            'sm_security_manager_group',
            self::OPTION_KEY,
            array($this, 'sanitize_options')
        );

        add_settings_section(
            'sm_security_main',
            'General Security Settings',
            function () {
                echo '<p>Baseline hardening and HTTP headers for this WordPress install.</p>';
            },
            'sm-security-manager'
        );

        $fields = array(
            'enable_headers'        => 'Enable security headers',
            'enable_csp'            => 'Enable Content Security Policy (CSP)',
            'disable_xmlrpc'        => 'Disable XML-RPC',
            'disable_file_edit'     => 'Disable file editor & plugin/theme installs',
            'block_author_enum'     => 'Block ?author=1 enumeration',
            'hide_wp_version'       => 'Hide WordPress version in front-end',
            'disable_feeds'         => 'Disable RSS/Atom feeds',
            'disable_emojis'        => 'Disable emojis',
            'disable_oembed'        => 'Disable oEmbed discovery routes',
            'disable_rest_users'    => 'Disable REST user endpoints for logged-out visitors',
            'disable_sitemaps'      => 'Disable core XML sitemaps',
            'enforce_strong_pw'     => 'Enforce strong passwords (admins/editors)',
            'hide_login_errors'     => 'Hide detailed login errors',
        );

        foreach ($fields as $key => $label) {
            add_settings_field(
                $key,
                $label,
                array($this, 'render_checkbox_field'),
                'sm-security-manager',
                'sm_security_main',
                array('key' => $key)
            );
        }

        add_settings_field(
            'csp_policy',
            'CSP Policy',
            array($this, 'field_csp_policy'),
            'sm-security-manager',
            'sm_security_main'
        );

        // Login section
        add_settings_section(
            'sm_security_login',
            'Login Protection',
            function () {
                echo '<p>Hide or protect the login page and limit brute-force attempts.</p>';
            },
            'sm-security-manager'
        );

        add_settings_field(
            'move_login_url_enabled',
            'Use custom login URL',
            array($this, 'render_checkbox_field'),
            'sm-security-manager',
            'sm_security_login',
            array('key' => 'move_login_url_enabled')
        );

        add_settings_field(
            'login_slug',
            'Custom login slug',
            array($this, 'field_login_slug'),
            'sm-security-manager',
            'sm_security_login'
        );

        add_settings_field(
            'rate_limit_enabled',
            'Enable login rate limiting',
            array($this, 'render_checkbox_field'),
            'sm-security-manager',
            'sm_security_login',
            array('key' => 'rate_limit_enabled')
        );

        add_settings_field(
            'rate_limit_settings',
            'Rate limit settings',
            array($this, 'field_rate_limit_settings'),
            'sm-security-manager',
            'sm_security_login'
        );

        // Turnstile
        add_settings_section(
            'sm_security_turnstile',
            'Cloudflare Turnstile',
            function () {
                echo '<p>Optional CAPTCHA protection on the login form using Cloudflare Turnstile.</p>';
            },
            'sm-security-manager'
        );

        add_settings_field(
            'turnstile_enabled',
            'Enable Turnstile on login',
            array($this, 'render_checkbox_field'),
            'sm-security-manager',
            'sm_security_turnstile',
            array('key' => 'turnstile_enabled')
        );

        add_settings_field(
            'turnstile_keys',
            'Turnstile keys',
            array($this, 'field_turnstile_keys'),
            'sm-security-manager',
            'sm_security_turnstile'
        );

        // Daily scan
        add_settings_section(
            'sm_security_scan',
            'Daily Update Scan',
            function () {
                echo '<p>Daily check for core, plugin and theme updates and email a summary.</p>';
            },
            'sm-security-manager'
        );

        add_settings_field(
            'daily_scan_enabled',
            'Enable daily scan email',
            array($this, 'render_checkbox_field'),
            'sm-security-manager',
            'sm_security_scan',
            array('key' => 'daily_scan_enabled')
        );

        add_settings_field(
            'daily_scan_email',
            'Scan email recipient',
            array($this, 'field_daily_scan_email'),
            'sm-security-manager',
            'sm_security_scan'
        );
    }

    public function sanitize_options($input) {
        $output = $this->get_options();

        $bool_keys = array(
            'enable_headers',
            'enable_csp',
            'disable_xmlrpc',
            'block_author_enum',
            'hide_wp_version',
            'disable_feeds',
            'disable_emojis',
            'disable_oembed',
            'disable_rest_users',
            'disable_sitemaps',
            'disable_file_edit',
            'enforce_strong_pw',
            'hide_login_errors',
            'move_login_url_enabled',
            'rate_limit_enabled',
            'turnstile_enabled',
            'daily_scan_enabled',
        );

        foreach ($bool_keys as $key) {
            $output[$key] = !empty($input[$key]) ? 1 : 0;
        }

        if (isset($input['csp_policy'])) {
            $output['csp_policy'] = sanitize_textarea_field($input['csp_policy']);
        }

        if (isset($input['login_slug'])) {
            $slug = sanitize_title_with_dashes($input['login_slug']);
            $output['login_slug'] = $slug ?: 'secure-login';
        }

        $output['rate_limit_max_attempts'] = isset($input['rate_limit_max_attempts']) ? max(1, (int) $input['rate_limit_max_attempts']) : 5;
        $output['rate_limit_window']       = isset($input['rate_limit_window']) ? max(60, (int) $input['rate_limit_window']) : 900;
        $output['rate_limit_lockout']      = isset($input['rate_limit_lockout']) ? max(60, (int) $input['rate_limit_lockout']) : 900;

        if (isset($input['turnstile_site_key'])) {
            $output['turnstile_site_key'] = sanitize_text_field($input['turnstile_site_key']);
        }
        if (isset($input['turnstile_secret_key'])) {
            $output['turnstile_secret_key'] = sanitize_text_field($input['turnstile_secret_key']);
        }

        if (isset($input['daily_scan_email'])) {
            $email = sanitize_email($input['daily_scan_email']);
            if (!empty($email)) {
                $output['daily_scan_email'] = $email;
            }
        }

        return $output;
    }

    /* ----- Field renderers ----- */

    public function render_checkbox_field($args) {
        $key  = $args['key'];
        $opts = $this->get_options();
        ?>
        <label>
            <input type="checkbox"
                   name="<?php echo esc_attr(self::OPTION_KEY . '[' . $key . ']'); ?>"
                   value="1" <?php checked(!empty($opts[$key]), 1); ?>>
        </label>
        <?php
    }

    public function field_csp_policy() {
        $opts = $this->get_options();
        ?>
        <textarea
            name="<?php echo esc_attr(self::OPTION_KEY); ?>[csp_policy]"
            rows="6"
            cols="80"
        ><?php echo esc_textarea($opts['csp_policy']); ?></textarea>
        <p class="description">
            Default policy is Gutenberg/Elementor-friendly and supports Google Fonts, GA4/GTM and Cloudflare Insights.
            Only change if you know what you’re doing.
        </p>
        <?php
    }

    public function field_login_slug() {
        $opts = $this->get_options();
        ?>
        <input type="text"
               name="<?php echo esc_attr(self::OPTION_KEY); ?>[login_slug]"
               value="<?php echo esc_attr($opts['login_slug']); ?>"
               class="regular-text">
        <p class="description">
            This becomes <code><?php echo esc_html(home_url('/')); ?><strong><?php echo esc_html($opts['login_slug']); ?></strong>/</code> as the login URL when enabled.
            After changing, you should save and then go to Settings → Permalinks and click “Save” once, or just visit the site front-end to trigger a rewrite flush.
        </p>
        <?php
    }

    public function field_rate_limit_settings() {
        $opts = $this->get_options();
        ?>
        <label>
            Max attempts:
            <input type="number"
                   name="<?php echo esc_attr(self::OPTION_KEY); ?>[rate_limit_max_attempts]"
                   value="<?php echo (int) $opts['rate_limit_max_attempts']; ?>"
                   min="1"
                   style="width:80px;">
        </label>
        &nbsp;&nbsp;
        <label>
            Window (seconds):
            <input type="number"
                   name="<?php echo esc_attr(self::OPTION_KEY); ?>[rate_limit_window]"
                   value="<?php echo (int) $opts['rate_limit_window']; ?>"
                   min="60"
                   style="width:100px;">
        </label>
        &nbsp;&nbsp;
        <label>
            Lockout (seconds):
            <input type="number"
                   name="<?php echo esc_attr(self::OPTION_KEY); ?>[rate_limit_lockout]"
                   value="<?php echo (int) $opts['rate_limit_lockout']; ?>"
                   min="60"
                   style="width:100px;">
        </label>
        <p class="description">
            Defaults: 5 attempts in 15 minutes (900s), lockout for 15 minutes after hitting limit.
        </p>
        <?php
    }

    public function field_turnstile_keys() {
        $opts = $this->get_options();
        ?>
        <p>
            <label>
                Site key:<br>
                <input type="text"
                       name="<?php echo esc_attr(self::OPTION_KEY); ?>[turnstile_site_key]"
                       value="<?php echo esc_attr($opts['turnstile_site_key']); ?>"
                       class="regular-text">
            </label>
        </p>
        <p>
            <label>
                Secret key:<br>
                <input type="text"
                       name="<?php echo esc_attr(self::OPTION_KEY); ?>[turnstile_secret_key]"
                       value="<?php echo esc_attr($opts['turnstile_secret_key']); ?>"
                       class="regular-text">
            </label>
        </p>
        <p class="description">
            Get keys from your Cloudflare dashboard under <strong>Turnstile</strong>.
        </p>
        <?php
    }

    public function field_daily_scan_email() {
        $opts = $this->get_options();
        ?>
        <input type="email"
               name="<?php echo esc_attr(self::OPTION_KEY); ?>[daily_scan_email]"
               value="<?php echo esc_attr($opts['daily_scan_email']); ?>"
               class="regular-text">
        <p class="description">
            Daily summary of core/plugin/theme updates will be sent here (if enabled).
        </p>
        <?php
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap">
            <h1>Adam's Security Manager</h1>
            <p>Baseline security hardening and login protection for this WordPress site.</p>
            <form method="post" action="options.php">
                <?php
                settings_fields('sm_security_manager_group');
                do_settings_sections('sm-security-manager');
                submit_button();
                ?>
            </form>
            <hr>
            <p><em>Tip:</em> You can still stack Wordfence / Cloudflare WAF on top of this for brute-force and malware scanning.</p>
        </div>
        <?php
    }
}

SM_Security_Manager::get_instance();