<?php
/**
 * Plugin Name: Adam's SEO Manager
 * Description: Simple SEO helper with friendly “Adam’s Advice” for every post, plus basic site-wide SEO setup.
 * Author: Dori Media / Adam Crouchley
 * Version: 1.1.1
 * License: GPL2+
 * Text Domain: adams-seo-manager
 */

if (!defined('ABSPATH')) {
    exit;
}

class ASM_SEO_Manager {

    const OPTION_KEY = 'asm_seo_manager_options';

    const META_FOCUS    = '_asm_focus_keyword';
    const META_TITLE    = '_asm_seo_title';
    const META_DESC     = '_asm_meta_description';
    const META_SCORE    = '_asm_seo_score';
    const META_NOINDEX  = '_asm_robots_noindex';
    const META_NOFOLLOW = '_asm_robots_nofollow';

    private static $instance = null;

    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        register_activation_hook(__FILE__, array($this, 'activate'));

        // Meta box + saving
        add_action('add_meta_boxes', array($this, 'add_meta_box'));
        add_action('save_post', array($this, 'save_post_meta'), 10, 2);

        // Front-end SEO tags
        add_filter('pre_get_document_title', array($this, 'filter_document_title'), 20);
        add_action('wp_head', array($this, 'output_meta_tags'), 1);
        add_action('wp_head', array($this, 'output_open_graph_tags'), 5);
        add_action('wp_head', array($this, 'output_schema_org'), 10);

        // Robots.txt & sitemap
        add_filter('robots_txt', array($this, 'filter_robots_txt'), 10, 2);
        add_action('init', array($this, 'register_sitemap_rewrite'));
        add_action('template_redirect', array($this, 'handle_sitemap_request'));

        // Breadcrumb shortcode
        add_shortcode('adam_breadcrumbs', array($this, 'shortcode_breadcrumbs'));

        // Settings page
        if (is_admin()) {
            add_action('admin_menu', array($this, 'register_settings_page'));
            add_action('admin_init', array($this, 'register_settings'));

            // Columns: SEO score
            add_filter('manage_post_posts_columns', array($this, 'add_seo_score_column'));
            add_action('manage_post_posts_custom_column', array($this, 'render_seo_score_column'), 10, 2);

            add_filter('manage_page_posts_columns', array($this, 'add_seo_score_column'));
            add_action('manage_page_posts_custom_column', array($this, 'render_seo_score_column'), 10, 2);
        }
    }

    /* ========================================================
     * Activation / Options
     * ===================================================== */

    public function activate() {
        $defaults = array(
            'enable_title_format'     => 1,
            'title_format'            => '%post_title% | %site_name%',
            'enable_meta_description' => 1,
            'enable_canonical'        => 1,
            'enable_og'               => 1,
            'enable_twitter'          => 1,
            'brand_name'              => get_bloginfo('name'),
            'brand_logo'              => '',
            'default_meta_description'=> '',

            // Robots / sitemap
            'enable_sitemap'          => 1,
            'robots_block_staging'    => 0,
            'robots_custom'           => '',
        );

        $current = get_option(self::OPTION_KEY, array());
        update_option(self::OPTION_KEY, wp_parse_args($current, $defaults));

        // Register sitemap rewrite and flush
        $this->register_sitemap_rewrite();
        flush_rewrite_rules();
    }

    private function get_options() {
        $defaults = array(
            'enable_title_format'     => 1,
            'title_format'            => '%post_title% | %site_name%',
            'enable_meta_description' => 1,
            'enable_canonical'        => 1,
            'enable_og'               => 1,
            'enable_twitter'          => 1,
            'brand_name'              => get_bloginfo('name'),
            'brand_logo'              => '',
            'default_meta_description'=> '',
            'enable_sitemap'          => 1,
            'robots_block_staging'    => 0,
            'robots_custom'           => '',
        );
        $opts = get_option(self::OPTION_KEY, array());
        return wp_parse_args($opts, $defaults);
    }

    /* ========================================================
     * Meta box: Adam’s SEO Manager (per post)
     * ===================================================== */

    public function add_meta_box() {
        $post_types = array('post', 'page'); // extend if needed

        foreach ($post_types as $pt) {
            add_meta_box(
                'asm_seo_box',
                "Adam's SEO Manager",
                array($this, 'render_meta_box'),
                $pt,
                'normal',
                'high'
            );
        }
    }

    public function render_meta_box($post) {
    wp_nonce_field('asm_seo_meta_save', 'asm_seo_meta_nonce');

    $focus      = get_post_meta($post->ID, self::META_FOCUS, true);
    $seo_title  = get_post_meta($post->ID, self::META_TITLE, true);
    $meta_desc  = get_post_meta($post->ID, self::META_DESC, true);
    $noindex    = (bool) get_post_meta($post->ID, self::META_NOINDEX, true);
    $nofollow   = (bool) get_post_meta($post->ID, self::META_NOFOLLOW, true);

    // Run analysis for Adam's Advice + SEO score
    $analysis = $this->analyse_post($post, $focus, $meta_desc);
    $advice   = $analysis['advice'];
    $score    = $analysis['score'];
    $suggest_internal = $analysis['internal_suggestions'];
    $image_hints      = $analysis['image_hints'];
    ?>
    <style>
        :root {
            --asm-bg: #f9fafb;
            --asm-border: #e5e7eb;
            --asm-muted: #6b7280;
            --asm-text: #111827;
            --asm-pill-ok: #fef3c7;
            --asm-pill-ok-text: #92400e;
            --asm-pill-good: #dcfce7;
            --asm-pill-good-text: #166534;
            --asm-pill-bad: #fee2e2;
            --asm-pill-bad-text: #b91c1c;
            --asm-score-good: #22c55e;
            --asm-score-ok: #fbbf24;
            --asm-score-bad: #ef4444;
            --asm-card-radius: 10px;
        }

        .asm-box {
            background: var(--asm-bg);
            border-radius: var(--asm-card-radius);
            border: 1px solid var(--asm-border);
            padding: 18px 18px 14px;
            margin: -6px -6px 0;
            box-shadow: 0 4px 8px rgba(15, 23, 42, 0.04);
            font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            color: var(--asm-text);
        }

        .asm-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 12px;
            margin-bottom: 12px;
            flex-wrap: wrap;
        }

        .asm-header-left {
            min-width: 200px;
        }

        .asm-title {
            font-weight: 600;
            font-size: 15px;
            margin: 0 0 4px;
        }

        .asm-subtitle {
            margin: 0;
            font-size: 12px;
            color: var(--asm-muted);
        }

        .asm-score-wrap {
            display: flex;
            flex-direction: column;
            align-items: flex-end;
            gap: 6px;
            min-width: 180px;
        }

        .asm-score-pill {
            display: inline-flex;
            align-items: center;
            gap: 6px;
            padding: 4px 10px;
            border-radius: 999px;
            font-weight: 600;
            font-size: 12px;
            white-space: nowrap;
        }

        .asm-score-pill span {
            font-weight: 500;
        }

        .asm-score-pill-good {
            background: var(--asm-pill-good);
            color: var(--asm-pill-good-text);
        }
        .asm-score-pill-ok {
            background: var(--asm-pill-ok);
            color: var(--asm-pill-ok-text);
        }
        .asm-score-pill-bad {
            background: var(--asm-pill-bad);
            color: var(--asm-pill-bad-text);
        }

        .asm-score-bar {
            position: relative;
            width: 180px;
            height: 6px;
            border-radius: 999px;
            background: #e5e7eb;
            overflow: hidden;
        }

        .asm-score-bar-inner {
            position: absolute;
            top: 0;
            left: 0;
            bottom: 0;
            border-radius: inherit;
            transition: width 0.3s ease;
        }

        .asm-score-bar-inner.good {
            background: linear-gradient(90deg, #4ade80, #16a34a);
        }
        .asm-score-bar-inner.ok {
            background: linear-gradient(90deg, #facc15, #f97316);
        }
        .asm-score-bar-inner.bad {
            background: linear-gradient(90deg, #fb7185, #ef4444);
        }

        .asm-layout {
            display: grid;
            grid-template-columns: minmax(0, 2fr) minmax(0, 1.4fr);
            gap: 20px;
            margin-top: 12px;
        }

        @media (max-width: 1024px) {
            .asm-layout {
                grid-template-columns: 1fr;
            }
            .asm-score-wrap {
                align-items: flex-start;
            }
        }

        .asm-panel {
            background: #ffffff;
            border-radius: 8px;
            border: 1px solid #e5e7eb;
            padding: 14px 14px 10px;
        }

        .asm-panel h3 {
            margin: 0 0 6px;
            font-size: 13px;
            font-weight: 600;
        }

        .asm-panel p.description {
            margin: 0 0 10px;
            font-size: 11px;
            color: var(--asm-muted);
        }

        .asm-field-row {
            margin-bottom: 10px;
        }

        .asm-field-row label {
            font-weight: 500;
            font-size: 12px;
            display: block;
            margin-bottom: 3px;
        }

        .asm-input-text,
        .asm-input-textarea {
            width: 100%;
            max-width: 100%;
            border-radius: 6px;
            border: 1px solid #d1d5db;
            padding: 6px 8px;
            font-size: 13px;
            box-shadow: 0 1px 1px rgba(15, 23, 42, 0.02) inset;
        }

        .asm-input-textarea {
            min-height: 70px;
            resize: vertical;
        }

        .asm-field-row small {
            display: block;
            margin-top: 3px;
            font-size: 11px;
            color: var(--asm-muted);
        }

        .asm-robots-options label {
            font-weight: 400;
            font-size: 12px;
        }

        .asm-robots-options input {
            margin-right: 5px;
        }

        .asm-advice-list {
            margin-top: 4px;
        }

        .asm-advice-item {
            margin-bottom: 7px;
            padding: 7px 9px;
            border-radius: 6px;
            font-size: 12px;
            display: flex;
            align-items: flex-start;
            gap: 6px;
        }

        .asm-advice-item .asm-dot {
            width: 8px;
            height: 8px;
            border-radius: 999px;
            margin-top: 4px;
        }

        .asm-advice-good {
            background: #f0fdf4;
            border: 1px solid #bbf7d0;
        }
        .asm-advice-good .asm-dot { background: #16a34a; }

        .asm-advice-ok {
            background: #fffbeb;
            border: 1px solid #fed7aa;
        }
        .asm-advice-ok .asm-dot { background: #f97316; }

        .asm-advice-bad {
            background: #fef2f2;
            border: 1px solid #fecaca;
        }
        .asm-advice-bad .asm-dot { background: #ef4444; }

        .asm-advice-label {
            font-weight: 600;
            margin-right: 4px;
        }

        .asm-subsection {
            margin-top: 12px;
            border-top: 1px dashed #e5e7eb;
            padding-top: 8px;
        }

        .asm-subsection h4 {
            margin: 0 0 4px;
            font-size: 12px;
            font-weight: 600;
        }

        .asm-subsection ul {
            margin: 0;
            padding-left: 16px;
            font-size: 12px;
        }

        .asm-subsection li {
            margin-bottom: 3px;
        }

        .asm-tip {
            margin-top: 8px;
            font-size: 11px;
            color: var(--asm-muted);
        }
    </style>

    <div class="asm-box">
        <div class="asm-header">
            <div class="asm-header-left">
                <p class="asm-title">Adam's SEO Manager</p>
                <p class="asm-subtitle">
                    Fill these in, click <strong>Update</strong>, then follow Adam’s advice.
                </p>
            </div>

            <?php
            $score_class = 'asm-score-pill-ok';
            $bar_class   = 'ok';
            $score_label = 'Okay';
            if ($score >= 80) {
                $score_class = 'asm-score-pill-good';
                $bar_class   = 'good';
                $score_label = 'Great';
            } elseif ($score < 50) {
                $score_class = 'asm-score-pill-bad';
                $bar_class   = 'bad';
                $score_label = 'Needs work';
            }
            ?>
            <div class="asm-score-wrap">
                <div class="asm-score-pill <?php echo esc_attr($score_class); ?>">
                    <span>Adam’s verdict:</span> <?php echo esc_html($score_label); ?>
                    <span>· <?php echo (int) $score; ?>/100</span>
                </div>
                <div class="asm-score-bar" aria-hidden="true">
                    <div class="asm-score-bar-inner <?php echo esc_attr($bar_class); ?>"
                         style="width: <?php echo (int) max(5, $score); ?>%;"></div>
                </div>
            </div>
        </div>

        <div class="asm-layout">
            <!-- LEFT: Inputs -->
            <div class="asm-panel">
                <h3>Post basics</h3>
                <p class="description">
                    Tell search engines what this page is about. Keep it simple and human first.
                </p>

                <div class="asm-field-row">
                    <label for="asm_focus_keyword">Focus keyword or key phrase</label>
                    <input type="text" class="asm-input-text" id="asm_focus_keyword"
                           name="asm_focus_keyword"
                           placeholder="e.g. best hikes in Canterbury"
                           value="<?php echo esc_attr($focus); ?>">
                    <small>This is the main phrase someone types into Google to find this post.</small>
                </div>

                <div class="asm-field-row">
                    <label for="asm_seo_title">SEO title (optional)</label>
                    <input type="text" class="asm-input-text" id="asm_seo_title"
                           name="asm_seo_title"
                           placeholder="Leave empty to use the post title"
                           value="<?php echo esc_attr($seo_title); ?>">
                    <small>Try to keep it under ~60 characters and include your focus keyword.</small>
                </div>

                <div class="asm-field-row">
                    <label for="asm_meta_description">Meta description</label>
                    <textarea class="asm-input-textarea" id="asm_meta_description"
                              name="asm_meta_description"
                              placeholder="1–2 friendly sentences that make people want to click."><?php echo esc_textarea($meta_desc); ?></textarea>
                    <small>Imagine you’re inviting a friend to read this – around 120–155 characters works well.</small>
                </div>

                <div class="asm-field-row asm-robots-options">
                    <label>Robots options</label>
                    <label>
                        <input type="checkbox" name="asm_robots_noindex" value="1" <?php checked($noindex); ?>>
                        Ask search engines to ignore this page (<code>noindex</code>)
                    </label><br>
                    <label>
                        <input type="checkbox" name="asm_robots_nofollow" value="1" <?php checked($nofollow); ?>>
                        Discourage search engines from following links on this page (<code>nofollow</code>)
                    </label>
                    <small>Leave both unchecked for normal behaviour on public pages.</small>
                </div>
            </div>

            <!-- RIGHT: Adam’s Advice, internal links, images -->
            <div class="asm-panel">
                <h3>Adam’s Advice</h3>
                <p class="description">
                    These are plain-English suggestions. You don’t need every single one perfect – just aim for mostly green.
                </p>

                <div class="asm-advice-list">
                    <?php
                    if (empty($advice)) {
                        echo '<p>No advice yet – add some content and click Update.</p>';
                    } else {
                        foreach ($advice as $item) {
                            $status = $item['status'];
                            $class  = 'asm-advice-ok';
                            if ($status === 'good') {
                                $class = 'asm-advice-good';
                            } elseif ($status === 'bad') {
                                $class = 'asm-advice-bad';
                            }
                            echo '<div class="asm-advice-item ' . esc_attr($class) . '">';
                            echo '<span class="asm-dot"></span>';
                            echo '<div>';
                            echo '<span class="asm-advice-label">' . esc_html($item['label']) . '</span>';
                            echo esc_html($item['message']);
                            echo '</div>';
                            echo '</div>';
                        }
                    }
                    ?>
                    <p class="asm-tip">
                        Tip: after you make changes, click <strong>Update</strong> and check how your score and advice change.
                    </p>
                </div>

                <?php if (!empty($suggest_internal)) : ?>
                    <div class="asm-subsection">
                        <h4>Suggested internal links</h4>
                        <p class="description">Linking to these pages can help visitors (and Google) discover more of your content.</p>
                        <ul>
                            <?php foreach ($suggest_internal as $p) : ?>
                                <li>
                                    <a href="<?php echo esc_url(get_edit_post_link($p->ID)); ?>" target="_blank">
                                        <?php echo esc_html(get_the_title($p)); ?>
                                    </a>
                                    &nbsp;·&nbsp;
                                    <a href="<?php echo esc_url(get_permalink($p)); ?>" target="_blank">view live</a>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <?php if (!empty($image_hints)) : ?>
                    <div class="asm-subsection">
                        <h4>Image alt text suggestions</h4>
                        <p class="description">Alt text helps with accessibility and image search. Keep it short but descriptive.</p>
                        <ul>
                            <?php foreach ($image_hints as $hint) : ?>
                                <li>
                                    <strong><?php echo esc_html($hint['file']); ?></strong> – 
                                    Suggested alt: “<?php echo esc_html($hint['suggested']); ?>”
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php
}


    public function save_post_meta($post_id, $post) {
        if (!isset($_POST['asm_seo_meta_nonce']) || !wp_verify_nonce($_POST['asm_seo_meta_nonce'], 'asm_seo_meta_save')) {
            return;
        }
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Focus keyword
        if (isset($_POST['asm_focus_keyword'])) {
            $focus = sanitize_text_field(wp_unslash($_POST['asm_focus_keyword']));
            if ($focus !== '') {
                update_post_meta($post_id, self::META_FOCUS, $focus);
            } else {
                delete_post_meta($post_id, self::META_FOCUS);
            }
        }

        // SEO title
        if (isset($_POST['asm_seo_title'])) {
            $seo_title = sanitize_text_field(wp_unslash($_POST['asm_seo_title']));
            if ($seo_title !== '') {
                update_post_meta($post_id, self::META_TITLE, $seo_title);
            } else {
                delete_post_meta($post_id, self::META_TITLE);
            }
        }

        // Meta description
        if (isset($_POST['asm_meta_description'])) {
            $meta_desc = sanitize_textarea_field(wp_unslash($_POST['asm_meta_description']));
            if ($meta_desc !== '') {
                update_post_meta($post_id, self::META_DESC, $meta_desc);
            } else {
                delete_post_meta($post_id, self::META_DESC);
            }
        }

        // Robots options
        $noindex  = !empty($_POST['asm_robots_noindex']) ? 1 : 0;
        $nofollow = !empty($_POST['asm_robots_nofollow']) ? 1 : 0;

        if ($noindex) {
            update_post_meta($post_id, self::META_NOINDEX, 1);
        } else {
            delete_post_meta($post_id, self::META_NOINDEX);
        }

        if ($nofollow) {
            update_post_meta($post_id, self::META_NOFOLLOW, 1);
        } else {
            delete_post_meta($post_id, self::META_NOFOLLOW);
        }

        // Update analysis + score on save
        $focus      = get_post_meta($post_id, self::META_FOCUS, true);
        $meta_desc  = get_post_meta($post_id, self::META_DESC, true);
        $analysis   = $this->analyse_post($post, $focus, $meta_desc);
        update_post_meta($post_id, self::META_SCORE, (int) $analysis['score']);
    }

    /* ========================================================
     * Post analysis – Adam’s Advice + SEO score
     * ===================================================== */

    private function analyse_post(WP_Post $post, $focus_keyword, $meta_desc) {
        $advice = array();

        $title   = wp_strip_all_tags(get_the_title($post));
        $content = $post->post_content;
        $text    = wp_strip_all_tags($content);
        $text    = trim(preg_replace('/\s+/', ' ', $text));
        $word_count = ($text === '') ? 0 : str_word_count($text);

        $slug = $post->post_name;

        $focus = trim($focus_keyword);
        $focus_lower = strtolower($focus);

        /* ---------- Title checks ---------- */
        $len = mb_strlen($title);
        if ($len === 0) {
            $advice[] = array(
                'status' => 'bad',
                'label'  => 'Title',
                'message'=> 'You don’t have a title yet. Start with a clear, human-friendly headline.'
            );
        } elseif ($len < 30) {
            $advice[] = array(
                'status' => 'ok',
                'label'  => 'Title length',
                'message'=> 'Your title is very short. Try expanding it so people instantly know what the post is about.'
            );
        } elseif ($len > 65) {
            $advice[] = array(
                'status' => 'ok',
                'label'  => 'Title length',
                'message'=> 'Your title is quite long. It may get cut off in Google. If you can, trim it slightly.'
            );
        } else {
            $advice[] = array(
                'status' => 'good',
                'label'  => 'Title length',
                'message'=> 'Nice – your title length is in a good range for search results.'
            );
        }

        if ($focus !== '') {
            if (stripos($title, $focus) !== false) {
                $advice[] = array(
                    'status' => 'good',
                    'label'  => 'Keyword in title',
                    'message'=> 'Your focus keyword appears in the title. Great start.'
                );
            } else {
                $advice[] = array(
                    'status' => 'ok',
                    'label'  => 'Keyword in title',
                    'message'=> 'Try to work your focus keyword (or a close variation) into the title if it still sounds natural.'
                );
            }
        } else {
            $advice[] = array(
                'status' => 'ok',
                'label'  => 'Focus keyword',
                'message'=> 'You haven’t set a focus keyword yet. Add one above so we can give more targeted advice.'
            );
        }

        /* ---------- Content length / readability ---------- */
        if ($word_count === 0) {
            $advice[] = array(
                'status' => 'bad',
                'label'  => 'Content length',
                'message'=> 'There’s no content yet. Aim for at least a few paragraphs before publishing.'
            );
        } elseif ($word_count < 300) {
            $advice[] = array(
                'status' => 'ok',
                'label'  => 'Content length',
                'message'=> 'This is a short post. For most topics, aim for at least 300–600 words to cover the subject well.'
            );
        } else {
            $advice[] = array(
                'status' => 'good',
                'label'  => 'Content length',
                'message'=> 'Great – you’ve written a decent amount of content. Quality still matters more than length.'
            );
        }

        // Basic readability: average sentence length + paragraph count
        $sentences = preg_split('/[.!?]+(\s|$)/u', $text, -1, PREG_SPLIT_NO_EMPTY);
        $num_sentences = count($sentences);
        $avg_sentence = ($num_sentences > 0) ? $word_count / $num_sentences : 0;

        if ($avg_sentence > 30) {
            $advice[] = array(
                'status' => 'ok',
                'label'  => 'Readability',
                'message'=> 'Your sentences are quite long on average. Try mixing in some shorter sentences to make it easier to read.'
            );
        } else {
            $advice[] = array(
                'status' => 'good',
                'label'  => 'Readability',
                'message'=> 'Sentence length looks reasonable. As long as it sounds natural, you’re good.'
            );
        }

        $paragraphs = preg_split('/\n\s*\n/', trim($content));
        if (count($paragraphs) < 2 && $word_count > 200) {
            $advice[] = array(
                'status' => 'ok',
                'label'  => 'Paragraphs',
                'message'=> 'Consider breaking the content into a few shorter paragraphs so it’s not a big wall of text.'
            );
        }

        /* ---------- Keyword usage ---------- */
        if ($focus !== '' && $word_count > 0) {
            $count = substr_count(strtolower($text), $focus_lower);
            if ($count === 0) {
                $advice[] = array(
                    'status' => 'bad',
                    'label'  => 'Keyword usage',
                    'message'=> 'Your focus keyword isn’t mentioned in the content yet. Try to use it naturally a few times.'
                );
            } elseif ($count <= 8) {
                $advice[] = array(
                    'status' => 'good',
                    'label'  => 'Keyword usage',
                    'message'=> 'Your focus keyword shows up several times. As long as it reads naturally, you’re all good.'
                );
            } else {
                $advice[] = array(
                    'status' => 'ok',
                    'label'  => 'Keyword usage',
                    'message'=> 'You’ve used the focus keyword a lot. Double-check that the text still sounds natural and not “stuffed”.'
                );
            }

            $first_chunk = mb_substr($text, 0, 200);
            if (stripos($first_chunk, $focus) !== false) {
                $advice[] = array(
                    'status' => 'good',
                    'label'  => 'Keyword early in content',
                    'message'=> 'Nice – your focus keyword appears near the start of the content.'
                );
            } else {
                $advice[] = array(
                    'status' => 'ok',
                    'label'  => 'Keyword early in content',
                    'message'=> 'Try to mention your focus keyword in the first sentence or two if it feels natural.'
                );
            }
        }

        /* ---------- Headings ---------- */
        if (stripos($content, '<h2') !== false || stripos($content, '<h3') !== false) {
            $advice[] = array(
                'status' => 'good',
                'label'  => 'Headings',
                'message'=> 'You’re using headings (H2/H3). This helps readers and search engines understand the structure.'
            );
        } else {
            $advice[] = array(
                'status' => 'ok',
                'label'  => 'Headings',
                'message'=> 'Consider adding a couple of headings (H2/H3) to break up the content and make it easier to scan.'
            );
        }

        /* ---------- Images + alt ---------- */
        $image_hints = array();
        if (stripos($content, '<img') !== false) {
            preg_match_all('/<img[^>]*>/i', $content, $matches);
            $imgs = isset($matches[0]) ? $matches[0] : array();
            $missing_alt = 0;
            foreach ($imgs as $img_tag) {
                $has_alt = stripos($img_tag, 'alt=') !== false && !preg_match('/alt=["\']\s*["\']/', $img_tag);
                if (!$has_alt) {
                    $missing_alt++;
                }
                // Try to extract file name
                if (preg_match('/src=["\']([^"\']+)["\']/', $img_tag, $m)) {
                    $src = $m[1];
                    $file = basename(parse_url($src, PHP_URL_PATH));
                    $suggest = $focus !== '' ? $focus : preg_replace('/[-_]+/', ' ', preg_replace('/\.[^.]+$/', '', $file));
                    $image_hints[] = array(
                        'file'      => $file,
                        'suggested' => $suggest,
                    );
                }
            }

            if ($missing_alt === 0) {
                $advice[] = array(
                    'status' => 'good',
                    'label'  => 'Images',
                    'message'=> 'Nice – your images all have alt text. This helps with accessibility and image search.'
                );
            } else {
                $advice[] = array(
                    'status' => 'ok',
                    'label'  => 'Images',
                    'message'=> 'You have images, but some might be missing alt text. Add short descriptions so screen readers and Google know what they show.'
                );
            }
        } else {
            $advice[] = array(
                'status' => 'ok',
                'label'  => 'Images',
                'message'=> 'You don’t have any images yet. Adding at least one relevant image can make the page more engaging.'
            );
        }

        /* ---------- Links ---------- */
        preg_match_all('/<a[^>]+href=["\']([^"\']+)["\']/i', $content, $link_matches);
        $links = isset($link_matches[1]) ? $link_matches[1] : array();

        $home = home_url();
        $has_internal = false;
        $has_external = false;
        foreach ($links as $href) {
            if (strpos($href, $home) === 0 || (strpos($href, '/') === 0 && strpos($href, '//') !== 0)) {
                $has_internal = true;
            } elseif (strpos($href, 'http') === 0 && strpos($href, $home) !== 0) {
                $has_external = true;
            }
        }

        if ($has_internal) {
            $advice[] = array(
                'status' => 'good',
                'label'  => 'Internal links',
                'message'=> 'You link to other pages on your own site. That helps visitors and spreads authority internally.'
            );
        } else {
            $advice[] = array(
                'status' => 'ok',
                'label'  => 'Internal links',
                'message'=> 'Add at least one link to a related page on your own site. It keeps people exploring your content.'
            );
        }

        if ($has_external) {
            $advice[] = array(
                'status' => 'good',
                'label'  => 'External links',
                'message'=> 'You link out to other websites. That’s a natural sign you’re part of the wider web.'
            );
        } else {
            $advice[] = array(
                'status' => 'ok',
                'label'  => 'External links',
                'message'=> 'If it makes sense, link out to 1–2 helpful external resources. It can make your content more useful.'
            );
        }

        /* ---------- Slug ---------- */
        if ($focus !== '' && $slug) {
            if (stripos($slug, sanitize_title($focus)) !== false) {
                $advice[] = array(
                    'status' => 'good',
                    'label'  => 'URL slug',
                    'message'=> 'Your URL includes the focus keyword. Nice and clean.'
                );
            } else {
                $advice[] = array(
                    'status' => 'ok',
                    'label'  => 'URL slug',
                    'message'=> 'Consider editing the URL slug so it includes a short, clean version of your focus keyword.'
                );
            }
        }

        /* ---------- Meta description ---------- */
        if ($meta_desc !== '') {
            $desc_len = mb_strlen($meta_desc);
            if ($desc_len < 80) {
                $advice[] = array(
                    'status' => 'ok',
                    'label'  => 'Meta description',
                    'message'=> 'Your meta description is quite short. Try writing 1–2 fuller sentences to “sell” the click.'
                );
            } elseif ($desc_len > 170) {
                $advice[] = array(
                    'status' => 'ok',
                    'label'  => 'Meta description',
                    'message'=> 'Your meta description may be a bit long and could get cut off. See if you can tighten it slightly.'
                );
            } else {
                $advice[] = array(
                    'status' => 'good',
                    'label'  => 'Meta description',
                    'message'=> 'Great – your meta description is a nice length.'
                );
            }
        } else {
            $advice[] = array(
                'status' => 'ok',
                'label'  => 'Meta description',
                'message'=> 'You haven’t written a meta description yet. Add 1–2 friendly sentences that would make a stranger want to click.'
            );
        }

        /* ---------- Internal link suggestions ---------- */
        $internal_suggestions = array();
        if ($focus !== '') {
            $args = array(
                'post_type'      => array('post', 'page'),
                'post_status'    => 'publish',
                'posts_per_page' => 5,
                'post__not_in'   => array($post->ID),
                's'              => $focus,
            );
            $q = new WP_Query($args);
            if ($q->have_posts()) {
                foreach ($q->posts as $p) {
                    $internal_suggestions[] = $p;
                }
            }
            wp_reset_postdata();
        }

        /* ---------- SEO score ---------- */
        $score = $this->compute_score_from_advice($advice);

        return array(
            'advice'               => $advice,
            'score'                => $score,
            'internal_suggestions' => $internal_suggestions,
            'image_hints'          => $image_hints,
        );
    }

    private function compute_score_from_advice($advice) {
        if (empty($advice)) {
            return 0;
        }
        $max_points = 0;
        $points     = 0;

        foreach ($advice as $item) {
            $max_points += 2; // each item worth 0–2
            if ($item['status'] === 'good') {
                $points += 2;
            } elseif ($item['status'] === 'ok') {
                $points += 1;
            }
        }

        if ($max_points === 0) {
            return 0;
        }

        $score = round(($points / $max_points) * 100);
        return max(0, min(100, $score));
    }

    /* ========================================================
     * Front-end SEO tags
     * ===================================================== */

    public function filter_document_title($title) {
        $opts = $this->get_options();

        if (empty($opts['enable_title_format'])) {
            return $title;
        }

        if (!is_singular()) {
            $replacements = array(
                '%post_title%' => $title,
                '%site_name%'  => get_bloginfo('name'),
            );
            $formatted = strtr($opts['title_format'], $replacements);
            return $formatted;
        }

        global $post;
        if (!$post instanceof WP_Post) {
            return $title;
        }

        $custom_title = get_post_meta($post->ID, self::META_TITLE, true);
        $post_title   = $custom_title !== '' ? $custom_title : get_the_title($post);

        $replacements = array(
            '%post_title%' => wp_strip_all_tags($post_title),
            '%site_name%'  => get_bloginfo('name'),
        );

        $formatted = strtr($opts['title_format'], $replacements);
        return $formatted;
    }

    public function output_meta_tags() {
        if (is_feed()) {
            return;
        }

        $opts = $this->get_options();

        // Meta description
        if (!empty($opts['enable_meta_description'])) {
            $desc = '';

            if (is_singular()) {
                global $post;
                if ($post instanceof WP_Post) {
                    $meta = get_post_meta($post->ID, self::META_DESC, true);
                    if ($meta !== '') {
                        $desc = $meta;
                    } else {
                        $excerpt = $post->post_excerpt ? $post->post_excerpt : wp_trim_words(wp_strip_all_tags($post->post_content), 30);
                        $desc = $excerpt;
                    }
                }
            } elseif (is_home() || is_front_page()) {
                $desc = $opts['default_meta_description'];
            }

            $desc = trim($desc);
            if ($desc !== '') {
                echo '<meta name="description" content="' . esc_attr($desc) . '" />' . "\n";
            }
        }

        // Robots (noindex/nofollow per post)
        if (is_singular()) {
            global $post;
            if ($post instanceof WP_Post) {
                $noindex  = get_post_meta($post->ID, self::META_NOINDEX, true);
                $nofollow = get_post_meta($post->ID, self::META_NOFOLLOW, true);

                $directives = array();
                if ($noindex) {
                    $directives[] = 'noindex';
                }
                if ($nofollow) {
                    $directives[] = 'nofollow';
                }
                if (!empty($directives)) {
                    echo '<meta name="robots" content="' . esc_attr(implode(',', $directives)) . '" />' . "\n";
                }
            }
        }

        // Canonical
        if (!empty($opts['enable_canonical'])) {
            $canonical = '';
            if (is_singular()) {
                $canonical = get_permalink();
            } elseif (is_home() || is_front_page()) {
                $canonical = home_url('/');
            } elseif (is_category() || is_tag() || is_tax() || is_post_type_archive()) {
                $canonical = get_term_link(get_queried_object());
            } elseif (is_search()) {
                $canonical = get_search_link();
            }

            if ($canonical && !is_wp_error($canonical)) {
                echo '<link rel="canonical" href="' . esc_url($canonical) . '" />' . "\n";
            }
        }
    }

    public function output_open_graph_tags() {
        $opts = $this->get_options();
        if (empty($opts['enable_og']) && empty($opts['enable_twitter'])) {
            return;
        }

        global $post;

        $title = wp_get_document_title();
        $url   = (is_singular() && $post instanceof WP_Post) ? get_permalink($post) : home_url(add_query_arg(array(), $GLOBALS['wp']->request));
        $type  = (is_singular()) ? 'article' : 'website';

        // Description (reuse meta)
        $desc = '';
        if (is_singular() && $post instanceof WP_Post) {
            $meta = get_post_meta($post->ID, self::META_DESC, true);
            if ($meta !== '') {
                $desc = $meta;
            } else {
                $desc = wp_trim_words(wp_strip_all_tags($post->post_content), 30);
            }
        } else {
            $desc = $opts['default_meta_description'];
        }
        $desc = trim($desc);

        // Image (featured or brand logo)
        $image = '';
        if (is_singular() && $post instanceof WP_Post && has_post_thumbnail($post)) {
            $img = wp_get_attachment_image_src(get_post_thumbnail_id($post), 'full');
            if (!empty($img[0])) {
                $image = $img[0];
            }
        }
        if ($image === '' && !empty($opts['brand_logo'])) {
            $image = $opts['brand_logo'];
        }

        if (!empty($opts['enable_og'])) {
            echo '<meta property="og:title" content="' . esc_attr($title) . '" />' . "\n";
            echo '<meta property="og:type" content="' . esc_attr($type) . '" />' . "\n";
            echo '<meta property="og:url" content="' . esc_url($url) . '" />' . "\n";
            if ($desc !== '') {
                echo '<meta property="og:description" content="' . esc_attr($desc) . '" />' . "\n";
            }
            echo '<meta property="og:site_name" content="' . esc_attr(get_bloginfo('name')) . '" />' . "\n";
            if ($image !== '') {
                echo '<meta property="og:image" content="' . esc_url($image) . '" />' . "\n";
            }
        }

        if (!empty($opts['enable_twitter'])) {
            echo '<meta name="twitter:card" content="summary_large_image" />' . "\n";
            echo '<meta name="twitter:title" content="' . esc_attr($title) . '" />' . "\n";
            if ($desc !== '') {
                echo '<meta name="twitter:description" content="' . esc_attr($desc) . '" />' . "\n";
            }
            if ($image !== '') {
                echo '<meta name="twitter:image" content="' . esc_url($image) . '" />' . "\n";
            }
        }
    }

    public function output_schema_org() {
        $opts = $this->get_options();
        $brand = trim($opts['brand_name']);
        if ($brand === '') {
            return;
        }

        $data = array(
            '@context' => 'https://schema.org',
            '@type'    => 'WebSite',
            'name'     => $brand,
            'url'      => home_url('/'),
        );

        if (!empty($opts['brand_logo'])) {
            $data['publisher'] = array(
                '@type' => 'Organization',
                'name'  => $brand,
                'logo'  => array(
                    '@type' => 'ImageObject',
                    'url'   => $opts['brand_logo'],
                ),
            );
        }

        echo "<script type=\"application/ld+json\">\n" . wp_json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . "\n</script>\n";
    }

    /* ========================================================
     * Robots.txt & sitemap
     * ===================================================== */

    public function filter_robots_txt($output, $public) {
        $opts = $this->get_options();

        if (!empty($opts['robots_block_staging'])) {
            $base = "User-agent: *\nDisallow: /\n";
        } else {
            $base = "User-agent: *\nDisallow: /wp-admin/\nAllow: /wp-admin/admin-ajax.php\n";
        }

        if (!empty($opts['enable_sitemap'])) {
            $base .= 'Sitemap: ' . trailingslashit(home_url()) . "sitemap.xml\n";
        }

        $custom = trim($opts['robots_custom']);
        if ($custom !== '') {
            $base .= "\n" . $custom . "\n";
        }

        return $base;
    }

    public function register_sitemap_rewrite() {
        add_rewrite_tag('%asm_sitemap%', '1');
        add_rewrite_rule('sitemap\.xml$', 'index.php?asm_sitemap=1', 'top');
    }

    public function handle_sitemap_request() {
        if (get_query_var('asm_sitemap')) {
            $opts = $this->get_options();
            if (empty($opts['enable_sitemap'])) {
                status_header(404);
                echo 'Sitemap disabled.';
                exit;
            }

            header('Content-Type: application/xml; charset=utf-8');

            $posts = get_posts(array(
                'post_type'      => array('post', 'page'),
                'post_status'    => 'publish',
                'posts_per_page' => -1,
                'orderby'        => 'modified',
                'order'          => 'DESC',
            ));

            echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
            echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

            foreach ($posts as $p) {
                $loc = get_permalink($p);
                $mod = get_post_modified_time('c', true, $p);
                echo '<url>';
                echo '<loc>' . esc_url($loc) . '</loc>';
                echo '<lastmod>' . esc_html($mod) . '</lastmod>';
                echo '</url>' . "\n";
            }

            echo '</urlset>';
            exit;
        }
    }

    /* ========================================================
     * Breadcrumb shortcode
     * ===================================================== */

    public function shortcode_breadcrumbs() {
        if (is_front_page()) {
            return '';
        }

        $parts   = array();
        $parts[] = '<a href="' . esc_url(home_url('/')) . '">Home</a>';

        if (is_singular()) {
            global $post;
            if ($post instanceof WP_Post) {
                if ($post->post_type === 'post') {
                    $cat = get_the_category($post->ID);
                    if (!empty($cat)) {
                        $cat = $cat[0];
                        $parts[] = '<a href="' . esc_url(get_category_link($cat)) . '">' . esc_html($cat->name) . '</a>';
                    }
                }
                $parts[] = esc_html(get_the_title($post));
            }
        } elseif (is_category() || is_tag() || is_tax()) {
            $term = get_queried_object();
            if ($term && !is_wp_error($term)) {
                $parts[] = esc_html($term->name);
            }
        } elseif (is_search()) {
            $parts[] = 'Search: ' . esc_html(get_search_query());
        } elseif (is_archive()) {
            $parts[] = wp_strip_all_tags(get_the_archive_title());
        }

        return '<nav class="adam-breadcrumbs" aria-label="Breadcrumbs">' . implode(' &raquo; ', $parts) . '</nav>';
    }

    /* ========================================================
     * Settings page
     * ===================================================== */

    public function register_settings_page() {
        add_options_page(
            "Adam's SEO Manager",
            "Adam's SEO Manager",
            'manage_options',
            'asm-seo-manager',
            array($this, 'render_settings_page')
        );

        // Keyword overview as a submenu
        add_submenu_page(
            'asm-seo-manager',
            'Keyword Overview',
            'Keyword Overview',
            'manage_options',
            'asm-seo-keywords',
            array($this, 'render_keyword_overview_page')
        );
    }

    public function register_settings() {
        register_setting(
            'asm_seo_manager_group',
            self::OPTION_KEY,
            array($this, 'sanitize_options')
        );

        add_settings_section(
            'asm_seo_main',
            'Site-wide SEO Basics',
            function () {
                echo '<p>These settings give your whole site a solid SEO foundation. Keep it simple and human-friendly.</p>';
            },
            'asm-seo-manager'
        );

        $fields = array(
            'enable_title_format'     => 'Use custom title format',
            'enable_meta_description' => 'Output meta description tag',
            'enable_canonical'        => 'Output canonical URL tag',
            'enable_og'               => 'Open Graph tags (for Facebook & others)',
            'enable_twitter'          => 'Twitter Card tags',
            'enable_sitemap'          => 'Enable simple XML sitemap at /sitemap.xml',
            'robots_block_staging'    => 'Block all robots (good for staging sites)',
        );

        foreach ($fields as $key => $label) {
            add_settings_field(
                $key,
                $label,
                array($this, 'render_checkbox_field'),
                'asm-seo-manager',
                'asm_seo_main',
                array('key' => $key)
            );
        }

        add_settings_field(
            'title_format',
            'Title format',
            array($this, 'field_title_format'),
            'asm-seo-manager',
            'asm_seo_main'
        );

        add_settings_field(
            'default_meta_description',
            'Default site description',
            array($this, 'field_default_meta_description'),
            'asm-seo-manager',
            'asm_seo_main'
        );

        add_settings_section(
            'asm_seo_brand',
            'Brand details (for search & social)',
            function () {
                echo '<p>Used for Open Graph and basic Schema.org markup.</p>';
            },
            'asm-seo-manager'
        );

        add_settings_field(
            'brand_name',
            'Brand / site name',
            array($this, 'field_brand_name'),
            'asm-seo-manager',
            'asm_seo_brand'
        );

        add_settings_field(
            'brand_logo',
            'Logo URL',
            array($this, 'field_brand_logo'),
            'asm-seo-manager',
            'asm_seo_brand'
        );

        add_settings_section(
            'asm_seo_robots',
            'robots.txt rules',
            function () {
                echo '<p>These lines are appended to the generated robots.txt. Use carefully.</p>';
            },
            'asm-seo-manager'
        );

        add_settings_field(
            'robots_custom',
            'Custom robots.txt lines',
            array($this, 'field_robots_custom'),
            'asm-seo-manager',
            'asm_seo_robots'
        );
    }

    public function sanitize_options($input) {
        $output = $this->get_options();

        $bool_keys = array(
            'enable_title_format',
            'enable_meta_description',
            'enable_canonical',
            'enable_og',
            'enable_twitter',
            'enable_sitemap',
            'robots_block_staging',
        );

        foreach ($bool_keys as $key) {
            $output[$key] = !empty($input[$key]) ? 1 : 0;
        }

        if (isset($input['title_format'])) {
            $output['title_format'] = sanitize_text_field($input['title_format']);
        }

        if (isset($input['brand_name'])) {
            $output['brand_name'] = sanitize_text_field($input['brand_name']);
        }

        if (isset($input['brand_logo'])) {
            $output['brand_logo'] = esc_url_raw($input['brand_logo']);
        }

        if (isset($input['default_meta_description'])) {
            $output['default_meta_description'] = sanitize_textarea_field($input['default_meta_description']);
        }

        if (isset($input['robots_custom'])) {
            $output['robots_custom'] = sanitize_textarea_field($input['robots_custom']);
        }

        return $output;
    }

    /* ----- Settings field renderers ----- */

    public function render_checkbox_field($args) {
        $key  = $args['key'];
        $opts = $this->get_options();
        ?>
        <label>
            <input type="checkbox"
                   name="<?php echo esc_attr(self::OPTION_KEY . '[' . $key . ']'); ?>"
                   value="1" <?php checked(!empty($opts[$key]), 1); ?>>
        </label>
        <?php
    }

    public function field_title_format() {
        $opts = $this->get_options();
        ?>
        <input type="text"
               name="<?php echo esc_attr(self::OPTION_KEY); ?>[title_format]"
               value="<?php echo esc_attr($opts['title_format']); ?>"
               class="regular-text">
        <p class="description">
            Use <code>%post_title%</code> and <code>%site_name%</code>.<br>
            Example: <code>%post_title% | %site_name%</code>
        </p>
        <?php
    }

    public function field_default_meta_description() {
        $opts = $this->get_options();
        ?>
        <textarea name="<?php echo esc_attr(self::OPTION_KEY); ?>[default_meta_description]"
                  rows="3" cols="60"><?php echo esc_textarea($opts['default_meta_description']); ?></textarea>
        <p class="description">
            Short, friendly description of your site. Used on the homepage and as a fallback.
        </p>
        <?php
    }

    public function field_brand_name() {
        $opts = $this->get_options();
        ?>
        <input type="text"
               name="<?php echo esc_attr(self::OPTION_KEY); ?>[brand_name]"
               value="<?php echo esc_attr($opts['brand_name']); ?>"
               class="regular-text">
        <?php
    }

    public function field_brand_logo() {
        $opts = $this->get_options();
        ?>
        <input type="text"
               name="<?php echo esc_attr(self::OPTION_KEY); ?>[brand_logo]"
               value="<?php echo esc_attr($opts['brand_logo']); ?>"
               class="regular-text">
        <p class="description">
            Paste the full URL of a clean logo image (ideally a square PNG or SVG).
        </p>
        <?php
    }

    public function field_robots_custom() {
        $opts = $this->get_options();
        ?>
        <textarea name="<?php echo esc_attr(self::OPTION_KEY); ?>[robots_custom]"
                  rows="4" cols="60"><?php echo esc_textarea($opts['robots_custom']); ?></textarea>
        <p class="description">
            These lines will be added to the generated robots.txt file.
        </p>
        <?php
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap">
            <h1>Adam's SEO Manager</h1>
            <p>Friendly SEO guidance for humans. No jargon, just clear steps to make your content easier to find.</p>
            <form method="post" action="options.php">
                <?php
                settings_fields('asm_seo_manager_group');
                do_settings_sections('asm-seo-manager');
                submit_button();
                ?>
            </form>
            <hr>
            <p><em>Heads up:</em> If you’re also using another big SEO plugin (Yoast, Rank Math, etc.), you may want to turn off their title/meta features to avoid duplicates.</p>
        </div>
        <?php
    }

    /* ========================================================
     * Keyword overview screen
     * ===================================================== */

    public function render_keyword_overview_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $posts = get_posts(array(
            'post_type'      => array('post', 'page'),
            'post_status'    => array('publish', 'draft', 'pending', 'future'),
            'posts_per_page' => -1,
        ));
        ?>
        <div class="wrap">
            <h1>Keyword Overview – Adam's SEO Manager</h1>
            <p>Quick view of your focus keywords and SEO scores. Look for posts with no keyword or low scores to improve.</p>
            <table class="widefat striped">
                <thead>
                <tr>
                    <th>Title</th>
                    <th>Type</th>
                    <th>Status</th>
                    <th>Focus keyword</th>
                    <th>SEO score</th>
                </tr>
                </thead>
                <tbody>
                <?php if (empty($posts)) : ?>
                    <tr><td colspan="5">No posts or pages found.</td></tr>
                <?php else : ?>
                    <?php foreach ($posts as $p) :
                        $focus = get_post_meta($p->ID, self::META_FOCUS, true);
                        $score = get_post_meta($p->ID, self::META_SCORE, true);
                        if ($score === '') {
                            // lazily compute if missing
                            $meta_desc = get_post_meta($p->ID, self::META_DESC, true);
                            $analysis  = $this->analyse_post($p, $focus, $meta_desc);
                            $score     = $analysis['score'];
                            update_post_meta($p->ID, self::META_SCORE, (int) $score);
                        }
                        ?>
                        <tr>
                            <td>
                                <a href="<?php echo esc_url(get_edit_post_link($p->ID)); ?>">
                                    <?php echo esc_html(get_the_title($p)); ?>
                                </a>
                            </td>
                            <td><?php echo esc_html($p->post_type); ?></td>
                            <td><?php echo esc_html($p->post_status); ?></td>
                            <td><?php echo $focus ? esc_html($focus) : '<em>none</em>'; ?></td>
                            <td><?php echo (int) $score; ?>/100</td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    /* ========================================================
     * Admin columns – SEO score
     * ===================================================== */

    public function add_seo_score_column($columns) {
        $columns['asm_seo_score'] = 'SEO score';
        return $columns;
    }

    public function render_seo_score_column($column, $post_id) {
        if ($column !== 'asm_seo_score') {
            return;
        }
        $score = get_post_meta($post_id, self::META_SCORE, true);
        if ($score === '') {
            $score = 0;
        }
        $score = (int) $score;

        $color = '#ffb900';
        if ($score >= 80) {
            $color = '#46b450';
        } elseif ($score < 50) {
            $color = '#dc3232';
        }

        echo '<span style="font-weight:bold;color:' . esc_attr($color) . ';">' . $score . '</span>';
    }
}

ASM_SEO_Manager::get_instance();